/**
 * Simple Site Sync - Deploy JavaScript
 */
jQuery(document).ready(function($) {
    'use strict';
    
    // HTTPS環境チェック（localhostは許可）
    var isSecure = (location.protocol === 'https:') || 
                   (location.hostname === 'localhost') || 
                   (location.hostname === '127.0.0.1') ||
                   (location.hostname.indexOf('localhost:') === 0);
    
    if (!isSecure) {
        // HTTP環境では全てのデプロイ関連ボタンを無効化
        $('#sss-generate-api-key').prop('disabled', true);
        $('#sss-copy-api-key').prop('disabled', true);
        $('.sss-test-connection').prop('disabled', true);
        $('#sss-save-deploy-settings').prop('disabled', true);
        $('#sss-deploy-execute').prop('disabled', true);
        $('#sss-deploy-theme-pages-execute').prop('disabled', true);
        $('#sss-deploy-db-media-execute').prop('disabled', true);
        $('#sss-deploy-all-execute').prop('disabled', true);
        // 入力フィールドも無効化
        $('#deploy_target_url_1, #deploy_api_key_1, #deploy_target_url_2, #deploy_api_key_2').prop('disabled', true);
        $('#deploy_target_server').prop('disabled', true);
        $('#deploy_theme_for_pages').prop('disabled', true);
        $('.deploy-item, #deploy_include_db_media').prop('disabled', true);
        return; // 以降のイベントハンドラーを設定しない
    }
    
    // APIキー生成
    $('#sss-generate-api-key').on('click', function() {
        var newKey = 'sss_' + Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15);
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_save_api_key',
            api_key: newKey,
            nonce: sssAdmin.nonce
        }, function(response) {
            if (response.success) {
                $('#sss_local_api_key').val(newKey);
                alert('APIキーを生成しました');
            }
        });
    });
    
    // APIキーコピー
    $('#sss-copy-api-key').on('click', function() {
        var key = $('#sss_local_api_key').val();
        if (!key) {
            alert('APIキーがありません。先にキーを生成してください。');
            return;
        }
        navigator.clipboard.writeText(key).then(function() {
            alert('APIキーをコピーしました');
        });
    });
    
    // 設定保存
    $('#sss-save-deploy-settings').on('click', function() {
        var data = {
            action: 'sss_save_deploy_settings',
            server1_url: $('#deploy_target_url_1').val(),
            server1_api_key: $('#deploy_api_key_1').val(),
            server2_url: $('#deploy_target_url_2').val(),
            server2_api_key: $('#deploy_api_key_2').val(),
            nonce: sssAdmin.nonce
        };
        $.post(sssAdmin.ajaxUrl, data, function(response) {
            if (response.success) {
                alert('設定を保存しました');
            } else {
                alert('エラー: ' + response.data);
            }
        });
    });
    
    // 接続テスト（各サーバー）
    $('.sss-test-connection').on('click', function() {
        var $btn = $(this);
        var server = $btn.data('server');
        var $status = $('.sss-connection-status[data-server="' + server + '"]');
        
        var targetUrl = $('#deploy_target_url_' + server).val();
        var apiKey = $('#deploy_api_key_' + server).val();
        
        if (!targetUrl || !apiKey) {
            $status.html('<span style="color: #d63638;">✗ URLとAPIキーを入力してください</span>');
            return;
        }
        
        $btn.prop('disabled', true);
        $status.html('<span style="color: #666;">接続中...</span>');
        
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_test_connection',
            server: server,
            target_url: targetUrl,
            api_key: apiKey,
            nonce: sssAdmin.nonce
        }, function(response) {
            $btn.prop('disabled', false);
            if (response.success) {
                $status.html('<span style="color: #00a32a;">✓ 接続成功 (' + (response.data.site_name || 'OK') + ')</span>');
            } else {
                $status.html('<span style="color: #d63638;">✗ ' + response.data + '</span>');
            }
        }).fail(function() {
            $btn.prop('disabled', false);
            $status.html('<span style="color: #d63638;">✗ 接続エラー</span>');
        });
    });
    
    // チェックボックスとサーバー選択の変更時
    function updateDeployButton() {
        var hasCodeChecked = $('.deploy-item:checked').length > 0;
        var serverSelected = $('#deploy_target_server').val() !== '';
        var dbMediaChecked = $('#deploy_include_db_media').is(':checked');
        var hasAnyChecked = hasCodeChecked || dbMediaChecked;
        var themeForPagesSelected = $('#deploy_theme_for_pages').val() !== '';
        
        $('#sss-deploy-execute').prop('disabled', !(hasCodeChecked && serverSelected));
        $('#sss-deploy-theme-pages-execute').prop('disabled', !(themeForPagesSelected && serverSelected));
        $('#sss-deploy-db-media-execute').prop('disabled', !(dbMediaChecked && serverSelected));
        $('#sss-deploy-all-execute').prop('disabled', !(hasAnyChecked && serverSelected));
    }
    $('.deploy-item').on('change', updateDeployButton);
    $('#deploy_target_server').on('change', updateDeployButton);
    $('#deploy_include_db_media').on('change', updateDeployButton);
    $('#deploy_theme_for_pages').on('change', updateDeployButton);
    
    // オーバーレイ表示/非表示
    function showDeployOverlay() {
        $('#sss-deploy-overlay').fadeIn(300);
        $('#sss-overlay-progress').css('width', '0%');
        $('#sss-overlay-status').text('準備中...');
        $(window).on('beforeunload.sss', function() {
            return '同期処理中です。ページを離れると処理が中断される可能性があります。';
        });
    }
    function hideDeployOverlay() {
        $('#sss-deploy-overlay').fadeOut(300);
        $(window).off('beforeunload.sss');
        $(window).off('beforeunload');
        window.onbeforeunload = null;
    }
    function updateOverlay(progress, status) {
        $('#sss-overlay-progress').css('width', progress + '%');
        $('#sss-overlay-status').text(status);
    }
    
    // ログ追加関数
    function addLog(message, type) {
        var $log = $('#sss-deploy-log');
        var color = type === 'error' ? '#f44336' : (type === 'success' ? '#4caf50' : '#d4d4d4');
        $log.append('<div style="color: ' + color + ';">[' + new Date().toLocaleTimeString() + '] ' + message + '</div>');
        $log.scrollTop($log[0].scrollHeight);
    }
    
    // コード反映実行
    $('#sss-deploy-execute').on('click', function() {
        var server = $('#deploy_target_server').val();
        var serverName = server === '1' ? 'テスト環境' : '本番環境';
        
        if (!confirm('選択したコンテンツを「' + serverName + '」に同期します。よろしいですか？')) {
            return;
        }
        
        var themes = [];
        var plugins = [];
        $('input[name="deploy_themes[]"]:checked').each(function() {
            themes.push($(this).val());
        });
        $('input[name="deploy_plugins[]"]:checked').each(function() {
            plugins.push($(this).val());
        });
        
        var $btn = $(this);
        var $progress = $('#sss-deploy-progress');
        var $bar = $('#sss-progress-bar');
        var $text = $('#sss-progress-text');
        var $log = $('#sss-deploy-log');
        
        $btn.prop('disabled', true);
        $progress.show();
        $bar.css('width', '0%');
        $log.html('');
        showDeployOverlay();
        updateOverlay(10, 'コード同期中...');
        
        addLog('同期処理を開始します... (サーバー' + server + ': ' + serverName + ')', 'info');
        
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_deploy_files',
            server: server,
            themes: themes,
            plugins: plugins,
            nonce: sssAdmin.nonce
        }, function(response) {
            $btn.prop('disabled', false);
            hideDeployOverlay();
            if (response.success) {
                $bar.css('width', '100%');
                $text.text('完了');
                addLog('同期が完了しました！', 'success');
                if (response.data.log) {
                    response.data.log.forEach(function(msg) {
                        addLog(msg, 'info');
                    });
                }
            } else {
                $text.text('エラー');
                addLog('エラー: ' + response.data, 'error');
            }
            updateDeployButton();
        }).fail(function() {
            $btn.prop('disabled', false);
            hideDeployOverlay();
            $text.text('エラー');
            addLog('通信エラーが発生しました', 'error');
            updateDeployButton();
        });
    });
    
    // テーマ＋固定ページ同期実行
    $('#sss-deploy-theme-pages-execute').on('click', function() {
        var server = $('#deploy_target_server').val();
        var serverName = server === '1' ? 'テスト環境' : '本番環境';
        var themeSlug = $('#deploy_theme_for_pages').val();
        var themeName = $('#deploy_theme_for_pages option:selected').text();
        
        if (!themeSlug) {
            alert('テーマを選択してください');
            return;
        }
        
        if (!confirm('テーマ「' + themeName + '」と全固定ページを「' + serverName + '」に同期します。\n\n⚠️ 同期先で同じスラッグの固定ページは更新されます。\n⚠️ ページテンプレート設定も同期されます。\n\nよろしいですか？')) {
            return;
        }
        
        var $btn = $(this);
        var $progress = $('#sss-deploy-progress');
        var $bar = $('#sss-progress-bar');
        var $text = $('#sss-progress-text');
        var $log = $('#sss-deploy-log');
        
        $btn.prop('disabled', true);
        $progress.show();
        $bar.css('width', '0%');
        $log.html('');
        showDeployOverlay();
        updateOverlay(10, 'テーマ＋固定ページ同期中...');
        
        addLog('=== テーマ＋固定ページ同期を開始します ===', 'info');
        addLog('サーバー' + server + ': ' + serverName, 'info');
        addLog('テーマ: ' + themeName, 'info');
        
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_deploy_theme_with_pages',
            server: server,
            theme_slug: themeSlug,
            nonce: sssAdmin.nonce
        }, function(response) {
            $btn.prop('disabled', false);
            hideDeployOverlay();
            if (response.success) {
                $bar.css('width', '100%');
                $text.text('完了');
                addLog('', 'info');
                addLog('★★★ テーマ＋固定ページ同期が完了しました！ ★★★', 'success');
                if (response.data.log) {
                    response.data.log.forEach(function(msg) {
                        addLog(msg, 'info');
                    });
                }
            } else {
                $text.text('エラー');
                addLog('エラー: ' + response.data, 'error');
            }
            updateDeployButton();
        }).fail(function() {
            $btn.prop('disabled', false);
            hideDeployOverlay();
            $text.text('エラー');
            addLog('通信エラーが発生しました', 'error');
            updateDeployButton();
        });
    });
    
    // DB＆メディア反映実行（関数化）
    function executeDbMediaDeploy(server, serverName, callback) {
        var $bar = $('#sss-progress-bar');
        var $text = $('#sss-progress-text');
        
        addLog('DB＆メディア同期を開始します... (サーバー' + server + ': ' + serverName + ')', 'info');
        addLog('URL変換: ' + sssAdmin.homeUrl + ' → 同期先URL', 'info');
        
        // Step 1: データベース移行
        addLog('【Step 1/2】データベース同期中...', 'info');
        updateOverlay(30, 'データベース同期中...');
        
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_deploy_database',
            server_id: server,
            nonce: sssAdmin.nonce
        }).done(function(dbResponse) {
            if (dbResponse.success) {
                $bar.css('width', '50%');
                addLog('✓ データベース同期完了', 'success');
                if (dbResponse.data.message) {
                    addLog('  ' + dbResponse.data.message, 'info');
                }
                
                // Step 2: メディア同期
                addLog('【Step 2/2】メディアファイル同期中...', 'info');
                $text.text('メディア同期中...');
                updateOverlay(60, 'メディア同期中...');
                
                $.post(sssAdmin.ajaxUrl, {
                    action: 'sss_sync_media',
                    server_id: server,
                    nonce: sssAdmin.nonce
                }).done(function(mediaResponse) {
                    if (mediaResponse.success) {
                        addLog('✓ メディア同期完了', 'success');
                        addLog('★★★ DB＆メディア同期が完了しました！ ★★★', 'success');
                        callback(true);
                    } else {
                        addLog('✗ メディア同期エラー: ' + mediaResponse.data, 'error');
                        addLog('データベースは正常に同期されました', 'info');
                        callback(false, 'media_error');
                    }
                }).fail(function(xhr, status, error) {
                    addLog('✗ メディア同期通信エラー: ' + error, 'error');
                    addLog('データベースは正常に同期されました', 'info');
                    callback(false, 'media_error');
                });
            } else {
                addLog('✗ データベース同期エラー: ' + dbResponse.data, 'error');
                callback(false, 'db_error');
            }
        }).fail(function(xhr, status, error) {
            addLog('✗ データベース同期通信エラー: ' + error, 'error');
            callback(false, 'db_error');
        });
    }
    
    $('#sss-deploy-db-media-execute').on('click', function() {
        var server = $('#deploy_target_server').val();
        var serverName = server === '1' ? 'テスト環境' : '本番環境';
        
        if (!confirm('データベースとメディアファイルを「' + serverName + '」に同期します。\n\n⚠️ 同期先のデータ・ファイルは上書きされます。\n⚠️ ファイル数が多い場合は時間がかかります。\n\nよろしいですか？')) {
            return;
        }
        
        var $btn = $(this);
        var $progress = $('#sss-deploy-progress');
        var $bar = $('#sss-progress-bar');
        var $text = $('#sss-progress-text');
        var $log = $('#sss-deploy-log');
        
        $btn.prop('disabled', true);
        $progress.show();
        $bar.css('width', '0%');
        $log.html('');
        showDeployOverlay();
        updateOverlay(10, 'DB＆メディア同期準備中...');
        
        executeDbMediaDeploy(server, serverName, function(success, errorType) {
            $btn.prop('disabled', false);
            updateDeployButton();
            hideDeployOverlay();
            
            if (success) {
                $bar.css('width', '100%');
                $text.text('完了');
            } else if (errorType === 'media_error') {
                $bar.css('width', '75%');
                $text.text('メディア同期エラー（DB同期は完了）');
            } else {
                $text.text('エラー');
            }
        });
    });
    
    // 全て反映実行
    $('#sss-deploy-all-execute').on('click', function() {
        var server = $('#deploy_target_server').val();
        var serverName = server === '1' ? 'テスト環境' : '本番環境';
        
        var hasCode = $('.deploy-item:checked').length > 0;
        var hasDbMedia = $('#deploy_include_db_media').is(':checked');
        
        var confirmMsg = '選択した全ての項目を「' + serverName + '」に同期します。\n\n';
        if (hasCode) confirmMsg += '✓ コード（テーマ/プラグイン）\n';
        if (hasDbMedia) confirmMsg += '✓ データベース＆メディア\n';
        confirmMsg += '\n⚠️ 同期先のデータ・ファイルは上書きされます。\nよろしいですか？';
        
        if (!confirm(confirmMsg)) {
            return;
        }
        
        var themes = [];
        var plugins = [];
        $('input[name="deploy_themes[]"]:checked').each(function() {
            themes.push($(this).val());
        });
        $('input[name="deploy_plugins[]"]:checked').each(function() {
            plugins.push($(this).val());
        });
        
        var $btn = $(this);
        var $progress = $('#sss-deploy-progress');
        var $bar = $('#sss-progress-bar');
        var $text = $('#sss-progress-text');
        var $log = $('#sss-deploy-log');
        
        $btn.prop('disabled', true);
        $progress.show();
        $bar.css('width', '0%');
        $log.html('');
        showDeployOverlay();
        
        addLog('=== 全項目同期を開始します ===', 'info');
        addLog('サーバー' + server + ': ' + serverName, 'info');
        
        var steps = [];
        if (hasCode) steps.push('code');
        if (hasDbMedia) steps.push('dbmedia');
        var totalSteps = steps.length;
        var currentStep = 0;
        
        function runNextStep() {
            if (currentStep >= steps.length) {
                // 全て完了
                $btn.prop('disabled', false);
                updateDeployButton();
                hideDeployOverlay();
                $bar.css('width', '100%');
                $text.text('全て完了');
                addLog('', 'info');
                addLog('★★★ 全ての同期が完了しました！ ★★★', 'success');
                return;
            }
            
            var step = steps[currentStep];
            var baseProgress = (currentStep / totalSteps) * 100;
            
            if (step === 'code') {
                addLog('', 'info');
                addLog('【コード同期】', 'info');
                updateOverlay(baseProgress + 10, 'コード同期中...');
                
                $.post(sssAdmin.ajaxUrl, {
                    action: 'sss_deploy_files',
                    server: server,
                    themes: themes,
                    plugins: plugins,
                    nonce: sssAdmin.nonce
                }).done(function(response) {
                    if (response.success) {
                        addLog('✓ コード同期完了', 'success');
                        if (response.data.log) {
                            response.data.log.forEach(function(msg) {
                                addLog('  ' + msg, 'info');
                            });
                        }
                        currentStep++;
                        $bar.css('width', ((currentStep / totalSteps) * 100) + '%');
                        runNextStep();
                    } else {
                        addLog('✗ コード同期エラー: ' + response.data, 'error');
                        $btn.prop('disabled', false);
                        updateDeployButton();
                        hideDeployOverlay();
                        $text.text('エラー');
                    }
                }).fail(function() {
                    addLog('✗ コード同期通信エラー', 'error');
                    $btn.prop('disabled', false);
                    updateDeployButton();
                    hideDeployOverlay();
                    $text.text('エラー');
                });
            } else if (step === 'dbmedia') {
                addLog('', 'info');
                addLog('【DB＆メディア同期】', 'info');
                
                executeDbMediaDeploy(server, serverName, function(success, errorType) {
                    if (success) {
                        currentStep++;
                        $bar.css('width', ((currentStep / totalSteps) * 100) + '%');
                        runNextStep();
                    } else {
                        $btn.prop('disabled', false);
                        updateDeployButton();
                        hideDeployOverlay();
                        if (errorType === 'media_error') {
                            $text.text('メディア同期エラー（他は完了）');
                        } else {
                            $text.text('エラー');
                        }
                    }
                });
            }
        }
        
        runNextStep();
    });
    
    // デバッグログ更新
    $('#sss-refresh-log').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).text('更新中...');
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_get_debug_log',
            nonce: sssAdmin.nonce
        }, function(response) {
            $btn.prop('disabled', false).text('🔄 ログを更新');
            if (response.success) {
                $('#sss-debug-log').text(response.data);
                $('#sss-debug-log').scrollTop($('#sss-debug-log')[0].scrollHeight);
            }
        });
    });
    
    // デバッグログクリア
    $('#sss-clear-log').on('click', function() {
        if (!confirm('ログをクリアしますか？')) return;
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_clear_debug_log',
            nonce: sssAdmin.nonce
        }, function(response) {
            if (response.success) {
                $('#sss-debug-log').text('ログがクリアされました');
            }
        });
    });
    
    // 受信中フラグクリア
    $('#sss-clear-receiving-flag').on('click', function() {
        if (!confirm('受信中フラグを強制的にクリアしますか？\n\n※ 実際に同期処理中の場合は使用しないでください。')) return;
        var $btn = $(this);
        $btn.prop('disabled', true).text('処理中...');
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_clear_receiving_flag',
            nonce: sssAdmin.nonce
        }, function(response) {
            $btn.prop('disabled', false).html('⚠️ 受信中フラグをクリア');
            if (response.success) {
                alert('受信中フラグをクリアしました。ページを更新すると警告が消えます。');
                location.reload();
            } else {
                alert('エラー: ' + response.data);
            }
        }).fail(function() {
            $btn.prop('disabled', false).html('⚠️ 受信中フラグをクリア');
            alert('通信エラーが発生しました');
        });
    });

    // ライセンス有効化
    $('#sss-activate-license-btn').on('click', function() {
        var $btn = $(this);
        var key = $('#sss_license_key_field').val();
        
        if (!key) {
            alert('ライセンスキーを入力してください');
            return;
        }
        
        $btn.prop('disabled', true).text('処理中...');
        
        $.post(sssAdmin.ajaxUrl, {
            action: 'sss_activate_license',
            nonce: sssAdmin.nonce,
            license_key: key
        }, function(response) {
            $btn.prop('disabled', false).text('有効化する');
            if (response.success) {
                alert(response.data);
                location.reload();
            } else {
                alert('エラー: ' + response.data);
            }
        });
    });
});
