<?php
/**
 * Plugin Name: Simple Site Sync
 * Description: Simple SEO & Backup の拡張プラグイン。サイト間のリモートデプロイ・同期機能を追加します。
 * Version: 1.0.4
 * Author: XisCoreCreative Co., Ltd.
 * Author URI: https://simple-seo-migration.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: simple-site-sync
 * Domain Path: /languages
 * Requires Plugins: simple-seo-backup
 */

// 直接アクセス禁止
if (!defined('ABSPATH')) {
    exit;
}

// プラグイン定数
define('SSS_VERSION', '1.0.1');
define('SSS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SSS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SSS_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('SSS_FREE_SIZE_LIMIT', 25 * 1024 * 1024); // 無料版のサイズ制限: 25MB

// 配布サーバーURL（本番環境に合わせて変更）
if (!defined('SSS_UPDATE_SERVER')) {
    define('SSS_UPDATE_SERVER', 'https://simple-seo-migration.com');
}

// Plugin Update Checker ライブラリ読み込み
require_once SSS_PLUGIN_DIR . 'vendor/plugin-update-checker/plugin-update-checker.php';
use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

/**
 * Simple Site Sync メインクラス
 */
class Simple_Site_Sync {
    
    /**
     * シングルトンインスタンス
     */
    private static $instance = null;
    
    /**
     * コアプラグインインスタンス
     */
    private $core = null;
    
    /**
     * インスタンス取得
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * コンストラクタ
     */
    private function __construct() {
        // コアプラグインの存在チェック
        add_action('plugins_loaded', array($this, 'check_dependencies'));
    }
    
    /**
     * 依存関係チェック
     */
    public function check_dependencies() {
        if (!class_exists('Simple_SEO_Backup')) {
            add_action('admin_notices', array($this, 'show_dependency_notice'));
            return;
        }
        
        // コアプラグインのインスタンスを取得
        $this->core = Simple_SEO_Backup::get_instance();
        
        // 機能を初期化
        $this->init();
    }
    
    /**
     * 依存関係エラー通知
     */
    public function show_dependency_notice() {
        ?>
        <div class="notice notice-error">
            <p>
                <strong>Simple Site Sync</strong> には <strong>Simple SEO & Backup</strong> プラグインが必要です。
                先に Simple SEO & Backup をインストール・有効化してください。
            </p>
        </div>
        <?php
    }
    
    /**
     * 機能初期化
     */
    private function init() {
        // 管理画面
        add_action('admin_menu', array($this, 'add_admin_submenu'), 20);
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        
        // REST APIエンドポイント
        add_action('rest_api_init', array($this, 'register_deploy_endpoints'));
        
        // AJAXハンドラー
        add_action('wp_ajax_sss_deploy_files', array($this, 'ajax_deploy_files'));
        add_action('wp_ajax_sss_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_sss_deploy_database', array($this, 'ajax_deploy_database'));
        add_action('wp_ajax_sss_sync_media', array($this, 'ajax_sync_media'));
        add_action('wp_ajax_sss_save_api_key', array($this, 'ajax_save_api_key'));
        add_action('wp_ajax_sss_save_deploy_settings', array($this, 'ajax_save_deploy_settings'));
        add_action('wp_ajax_sss_clear_receiving_flag', array($this, 'ajax_clear_receiving_flag'));
        add_action('wp_ajax_sss_get_debug_log', array($this, 'ajax_get_debug_log'));
        add_action('wp_ajax_sss_clear_debug_log', array($this, 'ajax_clear_debug_log'));
        add_action('wp_ajax_sss_activate_license', array($this, 'ajax_activate_license'));
        add_action('wp_ajax_sss_test_error_report', array($this, 'ajax_test_error_report'));
        add_action('wp_ajax_sss_deploy_theme_with_pages', array($this, 'ajax_deploy_theme_with_pages'));
        
        // 受信中警告表示
        add_action('admin_notices', array($this, 'show_deploy_receiving_warning'));
        add_action('admin_footer', array($this, 'deploy_receiving_warning_script'));
    }
    
    /**
     * 管理画面サブメニュー追加
     */
    public function add_admin_submenu() {
        add_submenu_page(
            'simple-seo-backup',
            'サイト同期',
            '🔄 サイト同期',
            'manage_options',
            'simple-site-sync',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * 管理画面スクリプト読み込み
     */
    public function admin_enqueue_scripts($hook) {
        if (strpos($hook, 'simple-site-sync') === false) {
            return;
        }
        
        wp_enqueue_script('sss-admin', SSS_PLUGIN_URL . 'assets/deploy.js', array('jquery'), SSS_VERSION, true);
        wp_enqueue_style('sss-admin', SSS_PLUGIN_URL . 'assets/deploy.css', array(), SSS_VERSION);
        
        wp_localize_script('sss-admin', 'sssAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sss_deploy'),
            'homeUrl' => home_url(),
        ));
    }
    
    /**
     * ライセンスが有効かどうかを確認
     */
    public function is_license_active() {
        $token = get_option('sss_license_token', '');
        return !empty($token);
    }

    /**
     * サイズ制限チェック
     */
    public function check_size_limit($size) {
        if ($this->is_license_active()) {
            return true;
        }
        return $size <= SSS_FREE_SIZE_LIMIT;
    }

    /**
     * 設定ページ表示
     */
    public function render_settings_page() {
        $deploy_settings = get_option('sss_deploy_settings', array());
        $api_key = get_option('sss_api_key', '');
        $license_key = get_option('sss_license_key', '');
        $license_token = get_option('sss_license_token', '');
        
        $is_localhost = in_array($_SERVER['HTTP_HOST'] ?? '', array('localhost', '127.0.0.1'), true) 
                     || strpos($_SERVER['HTTP_HOST'] ?? '', 'localhost:') === 0;
        $is_secure = is_ssl() || $is_localhost;
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-cloud-upload"></span> Simple Site Sync</h1>
            
            <div class="sss-license-box <?php echo $license_token ? 'active' : ''; ?>" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin-bottom: 20px;">
                <h3 style="margin-top: 0;">🔑 ライセンス認証</h3>
                <div style="display: flex; gap: 10px; align-items: flex-end;">
                    <div style="flex: 1;">
                        <label class="description">マスターライセンスキーを入力してください</label><br>
                        <input type="text" id="sss_license_key_field" value="<?php echo esc_attr($license_key); ?>" class="regular-text" placeholder="ssm_...">
                    </div>
                    <button type="button" id="sss-activate-license-btn" class="button button-primary">
                        <?php echo $license_token ? 'ライセンス更新' : '有効化する'; ?>
                    </button>
                </div>
                <?php if ($license_token) : ?>
                    <p style="color: #00a32a; font-weight: bold; margin-top: 10px;">
                        <span class="dashicons dashicons-yes"></span> 制限解除済み（25MB制限なし）
                    </p>
                <?php else : ?>
                    <p style="color: #d63638; font-weight: bold; margin-top: 10px;">
                        <span class="dashicons dashicons-info"></span> 未有効化（25MBまでの制限あり）
                    </p>
                <?php endif; ?>
                <p class="description">キーは <a href="<?php echo esc_url(SSS_UPDATE_SERVER . '/dashboard/'); ?>" target="_blank">配布サイトのダッシュボード</a> から取得できます。</p>
            </div>

            <h1><span class="dashicons dashicons-cloud-upload"></span> Simple Site Sync</h1>
            <p class="description">テーマ・プラグイン・データベース・メディアを別のWordPressサイトにリモート同期します。</p>
            
            <?php if (!$is_secure) : ?>
            <div class="notice notice-error" style="padding: 15px;">
                <p style="margin: 0; font-weight: bold;">
                    <span class="dashicons dashicons-warning"></span>
                    HTTPS接続が必要です
                </p>
                <p style="margin: 10px 0 0 0;">
                    サイト同期機能はセキュリティ上の理由からHTTPS環境でのみ使用できます。
                </p>
            </div>
            <?php endif; ?>
            
            <div class="notice notice-info" style="padding: 15px;">
                <p style="margin: 0;"><strong>📌 使い方:</strong></p>
                <ol style="margin: 10px 0 0 20px;">
                    <li>同期先のWordPressにも「Simple SEO & Backup」と「Simple Site Sync」をインストール</li>
                    <li>同期先のプラグイン設定で「APIキー」を生成・コピー</li>
                    <li>こちらの設定画面で同期先URL・APIキーを入力</li>
                    <li>同期するコンテンツを選択して「同期実行」</li>
                </ol>
            </div>
            
            <h2>このサイトのAPI設定（受信用）</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">APIキー</th>
                    <td>
                        <div style="display: flex; gap: 10px; align-items: center;">
                            <input type="text" id="sss_local_api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text" readonly>
                            <button type="button" class="button" id="sss-generate-api-key">キーを生成</button>
                            <button type="button" class="button" id="sss-copy-api-key">コピー</button>
                        </div>
                        <p class="description">他のサイトからこのサイトにファイルを同期する際に必要なキー</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">受信エンドポイント</th>
                    <td>
                        <code><?php echo esc_url(home_url('?rest_route=/sss/v1/deploy')); ?></code>
                    </td>
                </tr>
            </table>
            
            <hr style="margin: 30px 0;">
            
            <h2>同期先サーバー設定（送信用）</h2>
            <p class="description">最大2つの同期先を設定できます（テスト環境・本番環境など）</p>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 20px;" class="sss-server-grid">
                <!-- サーバー1: テスト環境 -->
                <div style="background: #f0f6fc; padding: 20px; border-radius: 8px; border: 2px solid #0073aa;">
                    <h4 style="margin-top: 0; color: #0073aa;">
                        <span class="dashicons dashicons-desktop"></span>
                        サーバー1（テスト環境）
                    </h4>
                    <table class="form-table" style="margin: 0;">
                        <tr>
                            <th scope="row" style="padding: 10px 0;"><label for="deploy_target_url_1">URL</label></th>
                            <td style="padding: 10px 0;">
                                <input type="url" id="deploy_target_url_1" name="sss_deploy[server1][url]" 
                                       value="<?php echo esc_attr($deploy_settings['server1']['url'] ?? ''); ?>" 
                                       class="regular-text" placeholder="https://test.example.com">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row" style="padding: 10px 0;"><label for="deploy_api_key_1">APIキー</label></th>
                            <td style="padding: 10px 0;">
                                <input type="text" id="deploy_api_key_1" name="sss_deploy[server1][api_key]" 
                                       value="<?php echo esc_attr($deploy_settings['server1']['api_key'] ?? ''); ?>" 
                                       class="regular-text" placeholder="APIキー">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row" style="padding: 10px 0;">接続</th>
                            <td style="padding: 10px 0;">
                                <button type="button" class="button sss-test-connection" data-server="1">テスト</button>
                                <span class="sss-connection-status" data-server="1" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <!-- サーバー2: 本番環境 -->
                <div style="background: #fcf0f0; padding: 20px; border-radius: 8px; border: 2px solid #d63638;">
                    <h4 style="margin-top: 0; color: #d63638;">
                        <span class="dashicons dashicons-cloud"></span>
                        サーバー2（本番環境）
                    </h4>
                    <table class="form-table" style="margin: 0;">
                        <tr>
                            <th scope="row" style="padding: 10px 0;"><label for="deploy_target_url_2">URL</label></th>
                            <td style="padding: 10px 0;">
                                <input type="url" id="deploy_target_url_2" name="sss_deploy[server2][url]" 
                                       value="<?php echo esc_attr($deploy_settings['server2']['url'] ?? ''); ?>" 
                                       class="regular-text" placeholder="https://example.com">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row" style="padding: 10px 0;"><label for="deploy_api_key_2">APIキー</label></th>
                            <td style="padding: 10px 0;">
                                <input type="text" id="deploy_api_key_2" name="sss_deploy[server2][api_key]" 
                                       value="<?php echo esc_attr($deploy_settings['server2']['api_key'] ?? ''); ?>" 
                                       class="regular-text" placeholder="APIキー">
                            </td>
                        </tr>
                        <tr>
                            <th scope="row" style="padding: 10px 0;">接続</th>
                            <td style="padding: 10px 0;">
                                <button type="button" class="button sss-test-connection" data-server="2">テスト</button>
                                <span class="sss-connection-status" data-server="2" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <p style="margin-top: 20px;"><button type="button" class="button button-primary" id="sss-save-deploy-settings">設定を保存</button></p>
            
            <hr style="margin: 30px 0;">
            
            <h2>同期するコンテンツを選択</h2>
            
            <h3>テーマ</h3>
            <div style="background: #f9f9f9; padding: 15px; border-radius: 4px; max-height: 300px; overflow-y: auto;">
                <?php
                $themes = wp_get_themes();
                $current_theme = wp_get_theme();
                foreach ($themes as $theme_slug => $theme) :
                    $is_active = ($theme_slug === $current_theme->get_stylesheet());
                ?>
                <label style="display: block; padding: 8px; border-bottom: 1px solid #eee; <?php echo $is_active ? 'background: #e7f5ff;' : ''; ?>">
                    <input type="checkbox" name="deploy_themes[]" value="<?php echo esc_attr($theme_slug); ?>" class="deploy-item">
                    <strong><?php echo esc_html($theme->get('Name')); ?></strong>
                    <?php if ($is_active) : ?><span style="color: #0073aa; font-size: 11px;">(有効)</span><?php endif; ?>
                    <span style="color: #666; font-size: 12px; margin-left: 10px;"><?php echo esc_html($theme_slug); ?></span>
                </label>
                <?php endforeach; ?>
            </div>
            
            <hr style="margin: 30px 0;">
            
            <h3><span class="dashicons dashicons-welcome-widgets-menus" style="color: #9b59b6;"></span> テーマ＋固定ページ同期</h3>
            <p class="description">選択したテーマと、このサイトの全固定ページを一緒に同期します。ページテンプレートの設定も保持されます。</p>
            
            <div style="background: #f5f0ff; padding: 20px; border-radius: 8px; border: 1px solid #d4c4e8; margin-top: 15px;">
                <div style="margin-bottom: 15px;">
                    <label for="deploy_theme_for_pages" style="font-weight: 600; display: block; margin-bottom: 8px;">同期するテーマを選択:</label>
                    <select id="deploy_theme_for_pages" style="width: 100%; max-width: 400px; padding: 8px;">
                        <option value="">-- テーマを選択 --</option>
                        <?php foreach ($themes as $theme_slug => $theme) : ?>
                        <option value="<?php echo esc_attr($theme_slug); ?>">
                            <?php echo esc_html($theme->get('Name')); ?> (<?php echo esc_html($theme_slug); ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <p style="margin: 15px 0; color: #666; font-size: 12px;">
                    <strong>【同期される固定ページ】</strong><br>
                    <?php
                    $pages = get_posts(array(
                        'post_type' => 'page',
                        'post_status' => 'publish',
                        'posts_per_page' => -1,
                        'orderby' => 'menu_order',
                        'order' => 'ASC',
                    ));
                    if ($pages) :
                        foreach ($pages as $page) :
                            $template = get_page_template_slug($page->ID);
                            $template_name = $template ? basename($template) : 'default';
                    ?>
                    ✓ <?php echo esc_html($page->post_title); ?> <span style="color: #999;">(slug: <?php echo esc_html($page->post_name); ?>, template: <?php echo esc_html($template_name); ?>)</span><br>
                    <?php 
                        endforeach;
                    else :
                    ?>
                    <em>固定ページがありません</em>
                    <?php endif; ?>
                </p>
                
                <p style="margin: 10px 0 0 0; color: #9b59b6; font-size: 12px;">
                    ⚠️ 受信側で同じスラッグの固定ページが存在する場合は、コンテンツとテンプレート設定のみ更新されます（IDは保持）
                </p>
            </div>
            
            <h3 style="margin-top: 20px;">プラグイン</h3>
            <div style="background: #f9f9f9; padding: 15px; border-radius: 4px; max-height: 400px; overflow-y: auto;">
                <?php
                $plugins = get_plugins();
                $active_plugins = get_option('active_plugins', array());
                foreach ($plugins as $plugin_file => $plugin_data) :
                    $plugin_slug = dirname($plugin_file);
                    if ($plugin_slug === '.') $plugin_slug = basename($plugin_file, '.php');
                    $is_active = in_array($plugin_file, $active_plugins);
                ?>
                <label style="display: block; padding: 8px; border-bottom: 1px solid #eee; <?php echo $is_active ? 'background: #e7f5ff;' : ''; ?>">
                    <input type="checkbox" name="deploy_plugins[]" value="<?php echo esc_attr($plugin_file); ?>" class="deploy-item">
                    <strong><?php echo esc_html($plugin_data['Name']); ?></strong>
                    <?php if ($is_active) : ?><span style="color: #0073aa; font-size: 11px;">(有効)</span><?php endif; ?>
                    <span style="color: #666; font-size: 12px; margin-left: 10px;">v<?php echo esc_html($plugin_data['Version']); ?></span>
                </label>
                <?php endforeach; ?>
            </div>
            
            <hr style="margin: 30px 0;">
            
            <h3><span class="dashicons dashicons-database" style="color: #0073aa;"></span> データベース＆メディア同期</h3>
            <p class="description">データベースとメディアファイルを同期先サーバーに反映します。URLは自動的に変換されます。</p>
            
            <div style="background: #f0f6fc; padding: 20px; border-radius: 8px; border: 1px solid #c3c4c7; margin-top: 15px;">
                <label style="display: flex; align-items: center; margin-bottom: 15px;">
                    <input type="checkbox" id="deploy_include_db_media" name="deploy_include_db_media" value="1">
                    <span style="margin-left: 10px;"><strong>データベース＆メディアを同期する</strong></span>
                </label>
                <p style="margin: 0; color: #666; font-size: 12px;">
                    <strong>【データベース】</strong><br>
                    ✓ wp_options, wp_posts, wp_postmeta 等のWordPressテーブルを同期<br>
                    ✓ サイトURL（<?php echo esc_html(home_url()); ?>）は同期先URLに自動変換<br>
                    ✓ シリアライズされたデータ内のURLも正しく変換<br><br>
                    <strong>【メディアファイル】</strong><br>
                    ✓ wp-content/uploads/ 内の全ファイルを同期<br>
                    ⚠️ ファイル数が多い場合は時間がかかります
                </p>
            </div>
            
            <hr style="margin: 30px 0;">
            
            <div style="padding: 20px; background: #fff3cd; border-radius: 4px;">
                <p style="margin: 0 0 15px 0;"><strong>⚠️ 注意:</strong> 同期を実行すると、同期先の同名ファイル・データは上書きされます。</p>
                
                <div style="margin-bottom: 15px;">
                    <label for="deploy_target_server" style="font-weight: 600; margin-right: 10px;">同期先サーバー:</label>
                    <select id="deploy_target_server" style="padding: 8px 15px; font-size: 14px;">
                        <option value="">-- 選択してください --</option>
                        <option value="1" style="color: #0073aa;">サーバー1（テスト環境）</option>
                        <option value="2" style="color: #d63638;">サーバー2（本番環境）</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 15px; flex-wrap: wrap;">
                    <button type="button" class="button button-primary button-hero" id="sss-deploy-execute" disabled>
                        <span class="dashicons dashicons-cloud-upload" style="margin-top: 4px;"></span> コード同期
                    </button>
                    <button type="button" class="button button-hero" id="sss-deploy-theme-pages-execute" disabled style="background: #9b59b6; color: #fff; border-color: #9b59b6;">
                        <span class="dashicons dashicons-welcome-widgets-menus" style="margin-top: 4px;"></span> テーマ＋固定ページ同期
                    </button>
                    <button type="button" class="button button-hero" id="sss-deploy-db-media-execute" disabled style="background: #2271b1; color: #fff; border-color: #2271b1;">
                        <span class="dashicons dashicons-database" style="margin-top: 4px;"></span> DB＆メディア同期
                    </button>
                    <button type="button" class="button button-hero" id="sss-deploy-all-execute" disabled style="background: #135e96; color: #fff; border-color: #135e96;">
                        <span class="dashicons dashicons-migrate" style="margin-top: 4px;"></span> 全て同期
                    </button>
                </div>
                
                <span id="sss-deploy-status" style="display: block; margin-top: 15px;"></span>
            </div>
            
            <!-- 同期中オーバーレイ -->
            <div id="sss-deploy-overlay" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.8); z-index: 99999;">
                <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: #fff; padding: 40px; border-radius: 8px; text-align: center; max-width: 500px; width: 90%;">
                    <div style="font-size: 48px; margin-bottom: 20px;">🔄</div>
                    <h2 style="margin: 0 0 15px 0; color: #1d2327;">サイト同期中</h2>
                    <p style="color: #666; margin-bottom: 20px;">同期処理を実行しています。<br><strong style="color: #d63638;">このページを閉じたり、移動したりしないでください。</strong></p>
                    <div style="background: #eee; border-radius: 4px; height: 24px; overflow: hidden; margin-bottom: 15px;">
                        <div id="sss-overlay-progress" style="background: linear-gradient(90deg, #0073aa, #00a0d2); height: 100%; width: 0%; transition: width 0.3s;"></div>
                    </div>
                    <p id="sss-overlay-status" style="color: #666; font-size: 14px; margin: 0;">準備中...</p>
                </div>
            </div>
            
            <div id="sss-deploy-progress" style="display: none; margin-top: 20px;">
                <h4>同期進捗</h4>
                <div style="background: #eee; border-radius: 4px; height: 20px; overflow: hidden;">
                    <div id="sss-progress-bar" style="background: #0073aa; height: 100%; width: 0%; transition: width 0.3s;"></div>
                </div>
                <p id="sss-progress-text" style="margin-top: 10px; color: #666;"></p>
                <div id="sss-deploy-log" style="background: #1e1e1e; color: #d4d4d4; padding: 15px; border-radius: 4px; font-family: monospace; font-size: 12px; max-height: 300px; overflow-y: auto; margin-top: 10px;"></div>
            </div>
            
            <!-- デバッグログセクション -->
            <div style="margin-top: 30px; border-top: 1px solid #ddd; padding-top: 20px;">
                <h3>デバッグログ</h3>
                <p style="color: #666;">プラグインの詳細なログを確認できます。問題解決に役立ちます。</p>
                <button type="button" class="button" id="sss-refresh-log">🔄 ログを更新</button>
                <button type="button" class="button" id="sss-clear-log" style="margin-left: 10px;">🗑 ログをクリア</button>
                <button type="button" class="button" id="sss-clear-receiving-flag" style="margin-left: 10px; background: #f0ad4e; border-color: #eea236; color: #fff;">⚠️ 受信中フラグをクリア</button>
                <div id="sss-debug-log" style="background: #1e1e1e; color: #d4d4d4; padding: 15px; border-radius: 4px; font-family: monospace; font-size: 11px; max-height: 400px; overflow-y: auto; margin-top: 15px; white-space: pre-wrap;"><?php echo esc_html($this->get_log_contents(100)); ?></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * 受信中警告表示
     */
    public function show_deploy_receiving_warning() {
        $deploy_status = get_transient('sss_deploy_receiving');
        if ($deploy_status) {
            $type = isset($deploy_status['type']) ? $deploy_status['type'] : 'unknown';
            $started = isset($deploy_status['started']) ? $deploy_status['started'] : '';
            
            $type_label = '';
            switch ($type) {
                case 'theme': $type_label = 'テーマ'; break;
                case 'plugin': $type_label = 'プラグイン'; break;
                case 'database': $type_label = 'データベース'; break;
                case 'media': $type_label = 'メディアファイル'; break;
                default: $type_label = 'データ';
            }
            ?>
            <div class="notice notice-warning is-dismissible" id="sss-deploy-warning">
                <p style="font-size: 14px;">
                    <span class="dashicons dashicons-warning" style="color: #dba617;"></span>
                    <strong>⚠️ サイト更新中</strong> - 
                    現在、<?php echo esc_html($type_label); ?>の同期処理が進行中です。
                    <strong style="color: #d63638;">管理画面の操作やページ移動を行わないでください。</strong>
                    <?php if ($started): ?>
                    <br><small>開始時刻: <?php echo esc_html($started); ?></small>
                    <?php endif; ?>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * 受信中警告のJavaScript
     */
    public function deploy_receiving_warning_script() {
        $deploy_status = get_transient('sss_deploy_receiving');
        if ($deploy_status) {
            ?>
            <script>
            (function() {
                var warningActive = true;
                
                function checkDeployStatus() {
                    if (!warningActive) return;
                    
                    jQuery.get('<?php echo esc_url(home_url('?rest_route=/sss/v1/deploy-status')); ?>', function(response) {
                        if (!response.is_receiving) {
                            clearWarning();
                        }
                    }).fail(function() {
                        clearWarning();
                    });
                }
                
                function clearWarning() {
                    warningActive = false;
                    jQuery('#sss-deploy-warning').fadeOut(function() {
                        jQuery(this).remove();
                    });
                    window.onbeforeunload = null;
                }
                
                var checkInterval = setInterval(function() {
                    if (!warningActive) {
                        clearInterval(checkInterval);
                        return;
                    }
                    checkDeployStatus();
                }, 5000);
                
                setTimeout(checkDeployStatus, 2000);
                
                window.onbeforeunload = function() {
                    if (warningActive) {
                        return 'サイト更新中です。ページを離れると同期が失敗する可能性があります。';
                    }
                    return undefined;
                };
            })();
            </script>
            <?php
        }
    }
    
    /**
     * デプロイ受信状態を設定
     */
    private function set_deploy_receiving($type) {
        $time_str = function_exists('wp_date') ? wp_date('Y-m-d H:i:s') : current_time('Y-m-d H:i:s');
        set_transient('sss_deploy_receiving', array(
            'type' => $type,
            'started' => $time_str
        ), 1800);
    }
    
    /**
     * デプロイ受信状態をクリア
     */
    private function clear_deploy_receiving() {
        delete_transient('sss_deploy_receiving');
    }
    
    /**
     * REST APIエンドポイント登録
     */
    public function register_deploy_endpoints() {
        register_rest_route('sss/v1', '/deploy', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_deploy_receive'),
            'permission_callback' => array($this, 'verify_deploy_api_key'),
        ));
        
        register_rest_route('sss/v1', '/ping', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_ping'),
            'permission_callback' => array($this, 'verify_deploy_api_key'),
        ));
        
        register_rest_route('sss/v1', '/deploy-db', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_db_deploy_receive'),
            'permission_callback' => array($this, 'verify_deploy_api_key'),
        ));
        
        register_rest_route('sss/v1', '/deploy-media', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_media_deploy_receive'),
            'permission_callback' => array($this, 'verify_deploy_api_key'),
        ));
        
        register_rest_route('sss/v1', '/deploy-status', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_deploy_status'),
            'permission_callback' => '__return_true',
        ));

        register_rest_route('sss/v1', '/webhook/deactivate', array(
            'methods'  => 'POST',
            'callback' => array($this, 'handle_deactivate_webhook'),
            'permission_callback' => '__return_true',
        ));

        register_rest_route('sss/v1', '/report-error', array(
            'methods'  => 'POST',
            'callback' => array($this, 'handle_error_report'),
            'permission_callback' => '__return_true',
        ));
        
        // 固定ページ同期用エンドポイント
        register_rest_route('sss/v1', '/deploy-pages', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_pages_deploy_receive'),
            'permission_callback' => array($this, 'verify_deploy_api_key'),
        ));
    }
    
    /**
     * APIキー検証
     */
    public function verify_deploy_api_key($request) {
        $api_key = $request->get_header('X-SSS-API-Key');
        $stored_key = get_option('sss_api_key', '');
        
        if (empty($stored_key) || empty($api_key)) {
            return false;
        }
        
        return hash_equals($stored_key, $api_key);
    }
    
    /**
     * デプロイ状態確認ハンドラー
     */
    public function handle_deploy_status($request) {
        $deploy_status = get_transient('sss_deploy_receiving');
        return new WP_REST_Response(array(
            'is_receiving' => !empty($deploy_status),
            'type' => isset($deploy_status['type']) ? $deploy_status['type'] : null,
            'started' => isset($deploy_status['started']) ? $deploy_status['started'] : null,
        ));
    }
    
    /**
     * Pingハンドラー
     */
    public function handle_ping($request) {
        return new WP_REST_Response(array(
            'success' => true,
            'site_name' => get_bloginfo('name'),
            'site_url' => home_url(),
            'wp_version' => get_bloginfo('version'),
        ), 200);
    }
    
    /**
     * ログ出力
     */
    public function log($message, $level = 'info') {
        $log_file = SSS_PLUGIN_DIR . 'debug.log';
        
        // ファイルが存在しない場合は作成を試みる
        if (!file_exists($log_file)) {
            @touch($log_file);
            @chmod($log_file, 0666);
        }

        $timestamp = function_exists('wp_date') ? wp_date('Y-m-d H:i:s') : current_time('Y-m-d H:i:s');
        $level_upper = strtoupper($level);
        $log_line = "[{$timestamp}] [{$level_upper}] {$message}\n";
        
        $result = @file_put_contents($log_file, $log_line, FILE_APPEND | LOCK_EX);
        
        // 書き込みに失敗した場合はシステムログにも出力
        if ($result === false) {
            error_log("SSS Plugin Log Error: Failed to write to {$log_file}. Content: {$message}");
        }
    }
    
    /**
     * ログ取得
     */
    public function get_log_contents($lines = 100) {
        $log_file = SSS_PLUGIN_DIR . 'debug.log';
        
        if (!file_exists($log_file)) {
            return 'ログファイルがありません';
        }
        
        $content = file_get_contents($log_file);
        $all_lines = explode("\n", $content);
        $recent_lines = array_slice($all_lines, -$lines);
        
        return implode("\n", $recent_lines);
    }
    
    /**
     * ログクリア
     */
    public function clear_log() {
        $log_file = SSS_PLUGIN_DIR . 'debug.log';
        file_put_contents($log_file, '');
    }
    
    /**
     * PHP制限を解除
     */
    public function setup_unlimited_environment() {
        @ignore_user_abort(true);
        @set_time_limit(0);
        @ini_set('max_input_time', '-1');
        @ini_set('memory_limit', '-1');
        if (@ob_get_length()) {
            @ob_end_clean();
        }
    }
    
    /**
     * バイト数を人間が読みやすい形式に変換
     */
    public function format_bytes($bytes, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, $precision) . ' ' . $units[$pow];
    }
    
    /**
     * REST APIエンドポイントURL生成
     */
    private function build_rest_url($target_url, $route) {
        $base_url = trailingslashit($target_url);
        return $base_url . '?rest_route=' . $route;
    }
    
    /**
     * AJAX: APIキー保存
     */
    public function ajax_save_api_key() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        update_option('sss_api_key', $api_key);
        
        wp_send_json_success();
    }
    
    /**
     * AJAX: デプロイ設定保存
     */
    public function ajax_save_deploy_settings() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $settings = array(
            'server1' => array(
                'url' => isset($_POST['server1_url']) ? esc_url_raw(wp_unslash($_POST['server1_url'])) : '',
                'api_key' => isset($_POST['server1_api_key']) ? sanitize_text_field(wp_unslash($_POST['server1_api_key'])) : '',
            ),
            'server2' => array(
                'url' => isset($_POST['server2_url']) ? esc_url_raw(wp_unslash($_POST['server2_url'])) : '',
                'api_key' => isset($_POST['server2_api_key']) ? sanitize_text_field(wp_unslash($_POST['server2_api_key'])) : '',
            ),
        );
        
        update_option('sss_deploy_settings', $settings);
        
        wp_send_json_success();
    }
    
    /**
     * AJAX: 受信中フラグクリア
     */
    public function ajax_clear_receiving_flag() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $this->clear_deploy_receiving();
        $this->log('[手動] 受信中フラグをクリアしました');
        
        wp_send_json_success('受信中フラグをクリアしました');
    }
    
    /**
     * AJAX: デバッグログ取得
     */
    public function ajax_get_debug_log() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        wp_send_json_success($this->get_log_contents(200));
    }
    
    /**
     * AJAX: デバッグログクリア
     */
    public function ajax_clear_debug_log() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $this->clear_log();
        wp_send_json_success('ログをクリアしました');
    }
    
    /**
     * AJAX: 接続テスト
     */
    public function ajax_test_connection() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $target_url = isset($_POST['target_url']) ? esc_url_raw(wp_unslash($_POST['target_url'])) : '';
        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        
        if (empty($target_url) || empty($api_key)) {
            wp_send_json_error('URLとAPIキーを入力してください');
        }
        
        $endpoint = $this->build_rest_url($target_url, '/sss/v1/ping');
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 30,
            'headers' => array(
                'X-SSS-API-Key' => $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array('test' => true)),
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($code === 200 && isset($body['success'])) {
            wp_send_json_success(array(
                'site_name' => $body['site_name'] ?? '',
                'site_url' => $body['site_url'] ?? '',
            ));
        } elseif ($code === 401 || $code === 403) {
            wp_send_json_error('認証エラー: APIキーが正しくありません');
        } else {
            wp_send_json_error('接続エラー (HTTP ' . $code . ')');
        }
    }
    
    /**
     * AJAX: ファイル反映実行
     */
    public function ajax_deploy_files() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }

        // サイズチェック（送信側）
        $total_size = 0; // 実装の簡略化のため、実際には RecursiveDirectoryIterator 等で計算が必要
        // ... ここでサイズ計算ロジック ...
        
        // 受信側の制限チェック（PINGで確認するなどの拡張が可能）
        
        $server = isset($_POST['server']) ? sanitize_text_field(wp_unslash($_POST['server'])) : '';
        if (empty($server) || !in_array($server, array('1', '2'), true)) {
            wp_send_json_error('同期先サーバーを選択してください');
        }
        
        $server_key = 'server' . $server;
        $deploy_settings = get_option('sss_deploy_settings', array());
        
        $target_url = $deploy_settings[$server_key]['url'] ?? '';
        $api_key = $deploy_settings[$server_key]['api_key'] ?? '';
        
        if (empty($target_url) || empty($api_key)) {
            wp_send_json_error('サーバー' . $server . 'の設定が完了していません');
        }
        
        // テーマ名はURLエンコードされた日本語を含む可能性があるため、sanitize_text_fieldは使わない
        $themes = isset($_POST['themes']) ? wp_unslash((array) $_POST['themes']) : array();
        $plugins = isset($_POST['plugins']) ? array_map('sanitize_text_field', wp_unslash((array) $_POST['plugins'])) : array();
        
        if (empty($themes) && empty($plugins)) {
            wp_send_json_error('同期するコンテンツを選択してください');
        }
        
        $log = array();
        $endpoint = $this->build_rest_url($target_url, '/sss/v1/deploy');
        
        // テーマを反映
        foreach ($themes as $theme_slug) {
            // ディレクトリトラバーサル対策（..を除去）、スラッシュも除去
            $theme_slug = str_replace(array('..', '/', '\\'), '', $theme_slug);
            if (empty($theme_slug)) {
                $log[] = "テーマ名が不正です";
                continue;
            }
            $theme_dir = get_theme_root() . '/' . $theme_slug;
            
            if (!is_dir($theme_dir)) {
                $log[] = "テーマ {$theme_slug} が見つかりません";
                continue;
            }
            
            $zip_file = $this->create_zip($theme_dir, $theme_slug);
            if (!$zip_file) {
                $log[] = "テーマ {$theme_slug} のZIP作成に失敗";
                continue;
            }
            
            $result = $this->send_to_target($endpoint, 'theme', $theme_slug, $zip_file, $api_key);
            @unlink($zip_file);
            
            if ($result['success']) {
                $log[] = "✓ テーマ {$theme_slug} を同期しました";
            } else {
                $log[] = "✗ テーマ {$theme_slug}: " . $result['error'];
            }
        }
        
        // プラグインを反映
        foreach ($plugins as $plugin_file) {
            $plugin_file = sanitize_text_field($plugin_file);
            $plugin_slug = dirname($plugin_file);
            
            if ($plugin_slug === '.') {
                $plugin_path = WP_PLUGIN_DIR . '/' . $plugin_file;
                if (!file_exists($plugin_path)) {
                    $log[] = "プラグイン {$plugin_file} が見つかりません";
                    continue;
                }
                $zip_file = $this->create_zip_single_file($plugin_path, $plugin_file);
            } else {
                $plugin_dir = WP_PLUGIN_DIR . '/' . $plugin_slug;
                if (!is_dir($plugin_dir)) {
                    $log[] = "プラグイン {$plugin_slug} が見つかりません";
                    continue;
                }
                $zip_file = $this->create_zip($plugin_dir, $plugin_slug);
            }
            
            if (!$zip_file) {
                $log[] = "プラグイン {$plugin_file} のZIP作成に失敗";
                continue;
            }
            
            $result = $this->send_to_target($endpoint, 'plugin', $plugin_file, $zip_file, $api_key);
            @unlink($zip_file);
            
            if ($result['success']) {
                $log[] = "✓ プラグイン " . basename(dirname($plugin_file) === '.' ? $plugin_file : dirname($plugin_file)) . " を同期しました";
            } else {
                $log[] = "✗ プラグイン {$plugin_file}: " . $result['error'];
            }
        }
        
        wp_send_json_success(array('log' => $log));
    }
    
    /**
     * AJAX: テーマ + 固定ページ同期実行
     */
    public function ajax_deploy_theme_with_pages() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $server = isset($_POST['server']) ? sanitize_text_field(wp_unslash($_POST['server'])) : '';
        if (empty($server) || !in_array($server, array('1', '2'), true)) {
            wp_send_json_error('同期先サーバーを選択してください');
        }
        
        $server_key = 'server' . $server;
        $deploy_settings = get_option('sss_deploy_settings', array());
        
        $target_url = $deploy_settings[$server_key]['url'] ?? '';
        $api_key = $deploy_settings[$server_key]['api_key'] ?? '';
        
        if (empty($target_url) || empty($api_key)) {
            wp_send_json_error('サーバー' . $server . 'の設定が完了していません');
        }
        
        // テーマスラッグを取得（日本語対応）
        $theme_slug = isset($_POST['theme_slug']) ? wp_unslash($_POST['theme_slug']) : '';
        $theme_slug = str_replace(array('..', '/', '\\'), '', $theme_slug);
        
        if (empty($theme_slug)) {
            wp_send_json_error('テーマを選択してください');
        }
        
        $theme_dir = get_theme_root() . '/' . $theme_slug;
        if (!is_dir($theme_dir)) {
            wp_send_json_error("テーマ {$theme_slug} が見つかりません");
        }
        
        $log = array();
        
        // 1. テーマを同期
        $endpoint = $this->build_rest_url($target_url, '/sss/v1/deploy');
        
        $zip_file = $this->create_zip($theme_dir, $theme_slug);
        if (!$zip_file) {
            wp_send_json_error("テーマ {$theme_slug} のZIP作成に失敗");
        }
        
        $result = $this->send_to_target($endpoint, 'theme', $theme_slug, $zip_file, $api_key);
        @unlink($zip_file);
        
        if ($result['success']) {
            $log[] = "✓ テーマ {$theme_slug} を同期しました";
        } else {
            wp_send_json_error("テーマ同期失敗: " . $result['error']);
        }
        
        // 2. 固定ページデータを収集
        $pages = get_posts(array(
            'post_type' => 'page',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'orderby' => 'menu_order',
            'order' => 'ASC',
        ));
        
        if (empty($pages)) {
            $log[] = "⚠️ 同期する固定ページがありません";
            wp_send_json_success(array('log' => $log));
        }
        
        $pages_data = array();
        foreach ($pages as $page) {
            $template = get_page_template_slug($page->ID);
            $pages_data[] = array(
                'post_title'     => $page->post_title,
                'post_name'      => $page->post_name,
                'post_content'   => $page->post_content,
                'post_excerpt'   => $page->post_excerpt,
                'post_status'    => $page->post_status,
                'menu_order'     => $page->menu_order,
                'post_parent'    => $page->post_parent,
                'page_template'  => $template ? $template : '',
                'post_date'      => $page->post_date,
            );
        }
        
        // 3. 固定ページを送信
        $pages_endpoint = $this->build_rest_url($target_url, '/sss/v1/deploy-pages');
        $pages_result = $this->send_pages_to_target($pages_endpoint, $pages_data, $theme_slug, $api_key);
        
        if ($pages_result['success']) {
            $log[] = "✓ 固定ページ " . count($pages) . " 件を同期しました";
            if (!empty($pages_result['details'])) {
                foreach ($pages_result['details'] as $detail) {
                    $log[] = "  - " . $detail;
                }
            }
        } else {
            $log[] = "✗ 固定ページ同期失敗: " . $pages_result['error'];
        }
        
        wp_send_json_success(array('log' => $log));
    }
    
    /**
     * 固定ページデータを送信
     */
    private function send_pages_to_target($endpoint, $pages_data, $theme_slug, $api_key) {
        $this->log("[送信側] 固定ページ送信開始: " . count($pages_data) . " 件");
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 120,
            'headers' => array(
                'X-SSS-API-Key' => $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'pages' => $pages_data,
                'theme_slug' => $theme_slug,
            )),
        ));
        
        if (is_wp_error($response)) {
            $this->log("[送信側] 固定ページ送信エラー: " . $response->get_error_message(), 'error');
            return array('success' => false, 'error' => $response->get_error_message());
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($code !== 200) {
            $error_msg = $body['error'] ?? 'Unknown error (HTTP ' . $code . ')';
            $this->log("[送信側] 固定ページ送信失敗: {$error_msg}", 'error');
            return array('success' => false, 'error' => $error_msg);
        }
        
        $this->log("[送信側] 固定ページ送信完了", 'success');
        return array(
            'success' => true,
            'details' => $body['details'] ?? array(),
        );
    }
    
    /**
     * ディレクトリをZIP化
     */
    private function create_zip($source_dir, $name) {
        $zip_file = wp_tempnam('sss_zip_') . '.zip';
        
        $zip = new ZipArchive();
        if ($zip->open($zip_file, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            return false;
        }
        
        $exclude_patterns = array(
            'node_modules', '.git', '.svn', '.DS_Store', 'Thumbs.db',
            '.sass-cache', '.cache', '*.log', '*.sql', '*.bak',
        );
        
        $source_dir = realpath($source_dir);
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source_dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        $file_count = 0;
        foreach ($files as $file) {
            $file_path = $file->getRealPath();
            $relative_path = substr($file_path, strlen($source_dir) + 1);
            
            $exclude = false;
            foreach ($exclude_patterns as $pattern) {
                if (strpos($relative_path, $pattern . '/') !== false || 
                    strpos($relative_path, '/' . $pattern) !== false ||
                    basename($relative_path) === $pattern) {
                    $exclude = true;
                    break;
                }
                if (strpos($pattern, '*') !== false) {
                    $regex = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
                    if (preg_match($regex, basename($relative_path))) {
                        $exclude = true;
                        break;
                    }
                }
            }
            
            if ($exclude) {
                continue;
            }
            
            $zip_relative_path = $name . '/' . $relative_path;
            
            if (!mb_check_encoding($zip_relative_path, 'UTF-8')) {
                $zip_relative_path = mb_convert_encoding($zip_relative_path, 'UTF-8', 'auto');
            }
            
            if ($file->isDir()) {
                $zip->addEmptyDir($zip_relative_path);
            } else {
                if (defined('ZipArchive::FL_ENC_UTF_8')) {
                    $zip->addFile($file_path, $zip_relative_path, 0, 0, ZipArchive::FL_ENC_UTF_8);
                } else {
                    $zip->addFile($file_path, $zip_relative_path);
                }
                $file_count++;
            }
        }
        
        $zip->close();
        $this->log("ZIP作成完了: {$name} ({$file_count} ファイル)");
        
        return $zip_file;
    }
    
    /**
     * 単一ファイルをZIP化
     */
    private function create_zip_single_file($file_path, $name) {
        $zip_file = wp_tempnam('sss_zip_') . '.zip';
        
        $zip = new ZipArchive();
        if ($zip->open($zip_file, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            return false;
        }
        
        $zip->addFile($file_path, basename($file_path));
        $zip->close();
        
        return $zip_file;
    }
    
    /**
     * ターゲットサーバーにチャンク分割送信
     */
    private function send_to_target($endpoint, $type, $name, $zip_file, $api_key) {
        $this->setup_unlimited_environment();
        
        $zip_data = file_get_contents($zip_file);
        $zip_size = strlen($zip_data);
        
        $this->log("=== 送信開始 ===");
        $this->log("タイプ: {$type}, 名前: {$name}");
        $this->log("ZIPサイズ: " . number_format($zip_size) . " bytes");
        
        $chunk_size = 512 * 1024;
        $total_chunks = ceil($zip_size / $chunk_size);
        $session_id = wp_generate_uuid4();
        
        $this->log("チャンク分割: {$total_chunks} 個");
        
        for ($i = 0; $i < $total_chunks; $i++) {
            $offset = $i * $chunk_size;
            $chunk = substr($zip_data, $offset, $chunk_size);
            $base64_chunk = base64_encode($chunk);
            
            $is_last = ($i === $total_chunks - 1);
            
            if ($i > 0 && $i % 10 === 0) {
                usleep(100000);
            }
            
            $max_retries = 3;
            $response = null;
            for ($retry = 0; $retry < $max_retries; $retry++) {
                $response = wp_remote_post($endpoint, array(
                    'timeout' => 300,
                    'headers' => array(
                        'X-SSS-API-Key' => $api_key,
                        'Content-Type' => 'application/json',
                    ),
                    'body' => wp_json_encode(array(
                        'type' => $type,
                        'name' => $name,
                        'session_id' => $session_id,
                        'chunk_index' => $i,
                        'total_chunks' => $total_chunks,
                        'data' => $base64_chunk,
                        'is_last' => $is_last,
                    )),
                ));
                
                $code = wp_remote_retrieve_response_code($response);
                if (!is_wp_error($response) && $code !== 503) {
                    break;
                }
                
                if ($retry < $max_retries - 1) {
                    $this->log("チャンク " . ($i + 1) . " - 503エラー、リトライ...");
                    sleep(2);
                }
            }
            
            if (is_wp_error($response)) {
                return array('success' => false, 'error' => $response->get_error_message());
            }
            
            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            if ($code !== 200) {
                $error_msg = $body['error'] ?? 'Unknown error (HTTP ' . $code . ')';
                $this->log("チャンク {$i} 送信失敗: {$error_msg}", 'error');
                $this->report_error_to_server("send_to_target: {$type} ({$name})", "HTTP {$code}: {$error_msg}");
                return array('success' => false, 'error' => $error_msg);
            }
            
            $this->log("チャンク " . ($i + 1) . "/{$total_chunks} 送信成功");
        }
        
        $this->log("全チャンク送信完了", 'success');
        return array('success' => true);
    }

    /**
     * AJAX: ライセンス有効化
     */
    public function ajax_activate_license() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $license_key = isset($_POST['license_key']) ? sanitize_text_field(wp_unslash($_POST['license_key'])) : '';
        $domain = $_SERVER['HTTP_HOST'] ?? '';
        
        if (empty($license_key)) {
            wp_send_json_error('ライセンスキーを入力してください');
        }
        
        $response = wp_remote_post(SSS_UPDATE_SERVER . '/wp-json/ssm/v1/activate', array(
            'timeout' => 30,
            'body' => array(
                'license_key' => $license_key,
                'domain'      => $domain
            )
        ));
        
        if (is_wp_error($response)) {
            $error_msg = '配布サーバーとの通信に失敗しました: ' . $response->get_error_message();
            $this->report_error_to_server("ajax_activate_license", $error_msg);
            wp_send_json_error($error_msg);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);
        
        if (isset($body['success']) && $body['success']) {
            update_option('sss_license_key', $license_key);
            update_option('sss_license_token', $body['site_token']);
            wp_send_json_success('ライセンスを有効化しました');
        } else {
            $message = isset($body['message']) ? $body['message'] : '有効化に失敗しました';
            if ($code >= 400) {
                $this->report_error_to_server("ajax_activate_license", "HTTP {$code}: {$message}");
            }
            wp_send_json_error($message);
        }
    }

    /**
     * ライセンス無効化 Webhook
     */
    public function handle_deactivate_webhook($request) {
        $token = $request->get_param('site_token');
        $stored_token = get_option('sss_license_token', '');
        
        if (empty($token) || $token !== $stored_token) {
            return new WP_Error('invalid_token', '無効なトークンです', array('status' => 403));
        }
        
        delete_option('sss_license_token');
        // キー自体は残しておくが、トークンがないので制限がかかる
        
        return array('success' => true, 'message' => 'ライセンスを無効化しました');
    }

    /**
     * エラーレポート受信ハンドラー
     */
    public function handle_error_report($request) {
        $params = $request->get_json_params();
        $domain = sanitize_text_field($params['domain'] ?? '不明');
        $action = sanitize_text_field($params['action'] ?? '不明');
        $error  = sanitize_textarea_field($params['error'] ?? 'エラー詳細なし');
        $env    = $params['env'] ?? array();

        $log_msg = "\n[ERROR REPORT] ==========================\n";
        $log_msg .= "送信元ドメイン: {$domain}\n";
        $log_msg .= "実行アクション: {$action}\n";
        $log_msg .= "エラー内容: {$error}\n";
        $log_msg .= "環境情報:\n";
        foreach ($env as $key => $val) {
            $log_msg .= "  - " . sanitize_text_field($key) . ": " . sanitize_text_field($val) . "\n";
        }
        $log_msg .= "==========================================\n";

        $this->log($log_msg, 'error');

        return array('success' => true);
    }

    /**
     * AJAX: エラー報告テスト
     */
    public function ajax_test_error_report() {
        check_ajax_referer('sss_developer', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }

        $result = $this->report_error_to_server(
            'TEST_ACTION (Manual Test)', 
            'これはテストエラー通知です。500エラーのシミュレーションとして送信されました。',
            true // テストフラグ
        );

        if (is_wp_error($result)) {
            wp_send_json_error('送信失敗: ' . $result->get_error_message());
        }

        wp_send_json_success('テスト報告を送信しました。debug.log を確認してください。');
    }

    /**
     * エラーを配布サーバーに報告
     */
    private function report_error_to_server($action, $error_message, $test = false) {
        $domain = $_SERVER['HTTP_HOST'] ?? 'unknown';
        
        // 配布サーバー自身の場合は送信しない（テスト時を除く）
        if (!$test && ($domain === 'simple-seo-migration.com' || $domain === 'localhost')) {
            return;
        }

        $body = array(
            'domain' => $domain . ($test ? ' (TEST)' : ''),
            'action' => $action,
            'error'  => $error_message,
            'env'    => array(
                'php_version' => PHP_VERSION,
                'wp_version'  => get_bloginfo('version'),
                'sss_version' => SSS_VERSION,
            )
        );

        $args = array(
            'timeout'   => 15,
            'blocking'  => $test ? true : false, // テスト時は結果を確認するために同期送信
            'headers'   => array('Content-Type' => 'application/json'),
            'body'      => wp_json_encode($body),
            'sslverify' => false, // ループバック通信でのSSLエラーを回避
        );

        return wp_remote_post(SSS_UPDATE_SERVER . '/wp-json/sss/v1/report-error', $args);
    }

    /**
     * デプロイ受信ハンドラー
     */
    public function handle_deploy_receive($request) {
        try {
            $this->setup_unlimited_environment();
            
            $params = $request->get_json_params();
            $type = isset($params['type']) ? sanitize_text_field($params['type']) : 'code';
            $chunk_index = isset($params['chunk_index']) ? intval($params['chunk_index']) : 0;
            
            if ($chunk_index === 0) {
                $this->set_deploy_receiving($type);
            }
            
            $this->log("[受信側] === デプロイ受信開始 ===");
            
            if (!function_exists('wp_tempnam')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            
            if (empty($params['type']) || empty($params['name']) || empty($params['data'])) {
                $this->log("[受信側] エラー: パラメータ不足 (type=" . ($params['type']??'empty') . ", name=" . ($params['name']??'empty') . ")", 'error');
                return new WP_REST_Response(array('error' => 'Invalid request'), 400);
            }
            
            $type = sanitize_text_field($params['type']);
            $name = preg_replace('/[^a-zA-Z0-9\-_\/\.\%]/', '', $params['name']);
            $data = $params['data'];
            
            $session_id = isset($params['session_id']) ? sanitize_text_field($params['session_id']) : null;
            $chunk_index = isset($params['chunk_index']) ? intval($params['chunk_index']) : 0;
            $total_chunks = isset($params['total_chunks']) ? intval($params['total_chunks']) : 1;
            $is_last = ($chunk_index >= $total_chunks - 1);
            
            $chunk_data = base64_decode($data);
            if (!$chunk_data) {
                return new WP_REST_Response(array('error' => 'Invalid data encoding'), 400);
            }
            
            $temp_dir = sys_get_temp_dir();
            $temp_file = $temp_dir . '/sss_deploy_' . ($session_id ?: 'single') . '.zip';
            
            if ($chunk_index === 0) {
                $write_result = file_put_contents($temp_file, $chunk_data);
            } else {
                $write_result = file_put_contents($temp_file, $chunk_data, FILE_APPEND);
            }
            
            if ($write_result === false) {
                return new WP_REST_Response(array('error' => 'Failed to write temp file'), 500);
            }
            
            if (!$is_last) {
                return new WP_REST_Response(array(
                    'success' => true,
                    'message' => "チャンク " . ($chunk_index + 1) . "/{$total_chunks} 受信完了",
                ), 200);
            }
            
            // 最終チャンク処理
            $this->log("[受信側] 全チャンク受信完了、ZIP展開開始...");
            
            if ($type === 'theme') {
                $dest_dir = get_theme_root() . '/' . $name;
            } elseif ($type === 'plugin') {
                $plugin_dir = dirname($name);
                $dest_dir = WP_PLUGIN_DIR . '/' . $plugin_dir;
                if ($plugin_dir === '.' || $plugin_dir === '') {
                    $dest_dir = WP_PLUGIN_DIR;
                }
            } else {
                @unlink($temp_file);
                return new WP_REST_Response(array('error' => 'Invalid type'), 400);
            }
            
            if ($type === 'theme' || ($type === 'plugin' && dirname($name) !== '.' && dirname($name) !== '')) {
                if (is_dir($dest_dir)) {
                    $this->delete_directory($dest_dir);
                }
            }
            
            if (!function_exists('WP_Filesystem')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            WP_Filesystem();
            
            $extract_to = $type === 'theme' ? get_theme_root() : WP_PLUGIN_DIR;
            $result = unzip_file($temp_file, $extract_to);
            
            if (file_exists($temp_file)) {
                @unlink($temp_file);
            }
            
            if (is_wp_error($result)) {
                $this->log("[受信側] ZIP展開エラー: " . $result->get_error_message(), 'error');
                return new WP_REST_Response(array('error' => $result->get_error_message()), 500);
            }
            
            $this->log("[受信側] ★★★ デプロイ完了: {$name} ★★★", 'success');
            $this->clear_deploy_receiving();
            
            return new WP_REST_Response(array(
                'success' => true,
                'message' => $name . ' を同期しました',
            ), 200);
            
        } catch (Exception $e) {
            $this->log("[受信側] 例外発生: " . $e->getMessage(), 'error');
            $this->report_error_to_server("handle_deploy_receive: {$type} ({$name})", "Exception: " . $e->getMessage());
            $this->clear_deploy_receiving();
            return new WP_REST_Response(array('error' => 'Exception: ' . $e->getMessage()), 500);
        }
    }
    
    /**
     * 固定ページ受信ハンドラ
     */
    public function handle_pages_deploy_receive($request) {
        try {
            $params = $request->get_json_params();
            
            $this->log("[受信側] === 固定ページ受信開始 ===");
            
            if (empty($params['pages']) || !is_array($params['pages'])) {
                $this->log("[受信側] エラー: 固定ページデータがありません", 'error');
                return new WP_REST_Response(array('error' => 'No pages data'), 400);
            }
            
            $theme_slug = isset($params['theme_slug']) ? $params['theme_slug'] : '';
            $pages_data = $params['pages'];
            $details = array();
            
            // 親ページのスラッグからIDへのマッピングを作成（階層構造対応）
            $slug_to_id = array();
            
            // まず既存のページのマッピングを作成
            $existing_pages = get_posts(array(
                'post_type' => 'page',
                'post_status' => array('publish', 'draft', 'private'),
                'posts_per_page' => -1,
            ));
            foreach ($existing_pages as $ep) {
                $slug_to_id[$ep->post_name] = $ep->ID;
            }
            
            foreach ($pages_data as $page_data) {
                $post_name = sanitize_title($page_data['post_name']);
                $post_title = sanitize_text_field($page_data['post_title']);
                
                // 同じスラッグの固定ページを検索
                $existing_page = get_page_by_path($post_name);
                
                if ($existing_page) {
                    // 既存ページを更新（コンテンツとテンプレートのみ）
                    $update_data = array(
                        'ID'           => $existing_page->ID,
                        'post_content' => wp_kses_post($page_data['post_content']),
                        'post_excerpt' => sanitize_textarea_field($page_data['post_excerpt'] ?? ''),
                        'menu_order'   => intval($page_data['menu_order'] ?? 0),
                    );
                    
                    $result = wp_update_post($update_data, true);
                    
                    if (is_wp_error($result)) {
                        $details[] = "✗ {$post_title}: 更新失敗 - " . $result->get_error_message();
                        $this->log("[受信側] ページ更新失敗: {$post_title} - " . $result->get_error_message(), 'error');
                    } else {
                        // ページテンプレートを設定
                        if (!empty($page_data['page_template'])) {
                            update_post_meta($existing_page->ID, '_wp_page_template', $page_data['page_template']);
                        } else {
                            delete_post_meta($existing_page->ID, '_wp_page_template');
                        }
                        
                        $details[] = "✓ {$post_title}: 更新しました (ID: {$existing_page->ID})";
                        $this->log("[受信側] ページ更新成功: {$post_title} (ID: {$existing_page->ID})");
                        
                        // マッピングを更新
                        $slug_to_id[$post_name] = $existing_page->ID;
                    }
                } else {
                    // 新規ページを作成
                    $insert_data = array(
                        'post_type'    => 'page',
                        'post_title'   => $post_title,
                        'post_name'    => $post_name,
                        'post_content' => wp_kses_post($page_data['post_content']),
                        'post_excerpt' => sanitize_textarea_field($page_data['post_excerpt'] ?? ''),
                        'post_status'  => 'publish',
                        'menu_order'   => intval($page_data['menu_order'] ?? 0),
                    );
                    
                    // 親ページの処理（送信元の親ページスラッグから受信側のIDを取得）
                    // 注: 現在の実装では親ページ情報は送信していないため、0として設定
                    $insert_data['post_parent'] = 0;
                    
                    $new_page_id = wp_insert_post($insert_data, true);
                    
                    if (is_wp_error($new_page_id)) {
                        $details[] = "✗ {$post_title}: 作成失敗 - " . $new_page_id->get_error_message();
                        $this->log("[受信側] ページ作成失敗: {$post_title} - " . $new_page_id->get_error_message(), 'error');
                    } else {
                        // ページテンプレートを設定
                        if (!empty($page_data['page_template'])) {
                            update_post_meta($new_page_id, '_wp_page_template', $page_data['page_template']);
                        }
                        
                        $details[] = "✓ {$post_title}: 新規作成しました (ID: {$new_page_id})";
                        $this->log("[受信側] ページ作成成功: {$post_title} (ID: {$new_page_id})");
                        
                        // マッピングを更新
                        $slug_to_id[$post_name] = $new_page_id;
                    }
                }
            }
            
            $this->log("[受信側] ★★★ 固定ページ同期完了: " . count($pages_data) . " 件 ★★★", 'success');
            
            return new WP_REST_Response(array(
                'success' => true,
                'message' => count($pages_data) . ' 件の固定ページを同期しました',
                'details' => $details,
            ), 200);
            
        } catch (Exception $e) {
            $this->log("[受信側] 固定ページ同期例外: " . $e->getMessage(), 'error');
            return new WP_REST_Response(array('error' => 'Exception: ' . $e->getMessage()), 500);
        }
    }
    
    /**
     * ディレクトリ削除（再帰）
     */
    private function delete_directory($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->delete_directory($path);
            } else {
                @unlink($path);
            }
        }
        @rmdir($dir);
    }
    
    /**
     * ディレクトリ内のファイル数を再帰的にカウント
     */
    private function count_files_recursive($dir) {
        $count = 0;
        if (!is_dir($dir)) {
            return 0;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $count += $this->count_files_recursive($path);
            } else {
                $count++;
            }
        }
        return $count;
    }
    
    /**
     * AJAX: DBデプロイ
     */
    public function ajax_deploy_database() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $this->setup_unlimited_environment();
        
        $server_id = intval($_POST['server_id'] ?? 1);
        
        $deploy_settings = get_option('sss_deploy_settings', array());
        $server_key = 'server' . $server_id;
        
        if (empty($deploy_settings[$server_key]['url']) || empty($deploy_settings[$server_key]['api_key'])) {
            wp_send_json_error('ターゲットサーバーの設定が必要です');
        }
        
        $target_url = $deploy_settings[$server_key]['url'];
        $api_key = $deploy_settings[$server_key]['api_key'];
        
        $this->log("=== DB同期開始 ===");
        $this->log("送信元: " . home_url());
        $this->log("送信先: " . $target_url);
        
        global $wpdb;
        
        $source_url = home_url();
        $url_patterns = $this->build_url_replace_patterns($source_url, $target_url);
        
        $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);
        $table_prefix = $wpdb->prefix;
        
        $sql_file = wp_tempnam('sss_db_') . '.sql';
        $fp = fopen($sql_file, 'w');
        
        if (!$fp) {
            wp_send_json_error('一時ファイルの作成に失敗しました');
        }
        
        fwrite($fp, "-- Simple Site Sync Database Export\n");
        fwrite($fp, "SET FOREIGN_KEY_CHECKS=0;\n\n");
        
        $table_count = 0;
        $total_rows = 0;
        
        foreach ($tables as $table_row) {
            $table = $table_row[0];
            
            if (strpos($table, $table_prefix) !== 0) {
                continue;
            }
            
            $create_table = $wpdb->get_row("SHOW CREATE TABLE `{$table}`", ARRAY_N);
            if ($create_table) {
                fwrite($fp, "DROP TABLE IF EXISTS `{$table}`;\n");
                fwrite($fp, $create_table[1] . ";\n\n");
            }
            
            $offset = 0;
            $chunk_size = 500;
            
            while (true) {
                $safe_table = preg_replace('/[^a-zA-Z0-9_\-]/', '', $table);
                $rows = $wpdb->get_results(
                    $wpdb->prepare("SELECT * FROM `{$safe_table}` LIMIT %d OFFSET %d", $chunk_size, $offset),
                    ARRAY_A
                );
                
                if (empty($rows)) {
                    break;
                }
                
                foreach ($rows as $row) {
                    if ($table === $table_prefix . 'options') {
                        $option_name = isset($row['option_name']) ? $row['option_name'] : '';
                        if (in_array($option_name, array('sss_api_key', 'sss_deploy_settings', '_transient_sss_deploy_receiving'))) {
                            continue;
                        }
                    }
                    
                    $columns = array_keys($row);
                    $values = array();
                    
                    foreach ($row as $value) {
                        if (is_null($value)) {
                            $values[] = 'NULL';
                        } else {
                            $value = strtr($value, array_combine($url_patterns['old'], $url_patterns['new']));
                            $values[] = "'" . $wpdb->_real_escape($value) . "'";
                        }
                    }
                    
                    fwrite($fp, "INSERT INTO `{$table}` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $values) . ");\n");
                    $total_rows++;
                }
                
                $offset += $chunk_size;
            }
            
            fwrite($fp, "\n");
            $table_count++;
        }
        
        fwrite($fp, "SET FOREIGN_KEY_CHECKS=1;\n");
        fclose($fp);
        
        $zip_file = wp_tempnam('sss_db_') . '.zip';
        $zip = new ZipArchive();
        if ($zip->open($zip_file, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            @unlink($sql_file);
            wp_send_json_error('ZIP作成に失敗しました');
        }
        $zip->addFile($sql_file, 'database.sql');
        $zip->close();
        
        $endpoint = $this->build_rest_url($target_url, '/sss/v1/deploy-db');
        $result = $this->send_db_to_target($endpoint, $zip_file, $api_key, $table_prefix);
        
        @unlink($sql_file);
        @unlink($zip_file);
        
        if ($result['success']) {
            $this->log("★★★ DB同期完了 ★★★", 'success');
            wp_send_json_success(array(
                'message' => 'データベースを同期しました',
                'tables' => $table_count,
                'rows' => $total_rows,
            ));
        } else {
            $this->log("DB同期失敗: " . $result['error'], 'error');
            wp_send_json_error($result['error']);
        }
    }
    
    /**
     * URL置換パターン構築
     */
    public function build_url_replace_patterns($old_url, $new_url) {
        $old_values = array();
        $new_values = array();
        
        $old_url = untrailingslashit($old_url);
        $new_url = untrailingslashit($new_url);
        
        $old_urls_to_process = array($old_url);
        
        if (strpos($old_url, '//www.') !== false) {
            $old_www_inversion = str_replace('//www.', '//', $old_url);
        } else {
            $old_www_inversion = str_replace('//', '//www.', $old_url);
        }
        
        if ($old_www_inversion !== $old_url) {
            $old_urls_to_process[] = $old_www_inversion;
        }
        
        $old_schemes = array('http', 'https', '');
        $new_scheme = wp_parse_url($new_url, PHP_URL_SCHEME) ?: 'https';
        
        foreach ($old_urls_to_process as $old_url_variant) {
            foreach ($old_schemes as $old_scheme) {
                $old_with_scheme = $old_scheme ? preg_replace('/^https?/', $old_scheme, $old_url_variant) : preg_replace('/^https?:/', '', $old_url_variant);
                $new_with_scheme = $old_scheme ? preg_replace('/^https?/', $new_scheme, $new_url) : preg_replace('/^https?:/', '', $new_url);
                
                if (!in_array($old_with_scheme, $old_values)) {
                    $old_values[] = $old_with_scheme;
                    $new_values[] = $new_with_scheme;
                }
                
                $encoded_old = urlencode($old_with_scheme);
                if (!in_array($encoded_old, $old_values)) {
                    $old_values[] = $encoded_old;
                    $new_values[] = urlencode($new_with_scheme);
                }
                
                $json_escaped_old = addcslashes($old_with_scheme, '/');
                if (!in_array($json_escaped_old, $old_values)) {
                    $old_values[] = $json_escaped_old;
                    $new_values[] = addcslashes($new_with_scheme, '/');
                }
            }
        }
        
        return array('old' => $old_values, 'new' => $new_values);
    }
    
    /**
     * DBデータをチャンク送信
     */
    private function send_db_to_target($endpoint, $zip_file, $api_key, $table_prefix) {
        $this->setup_unlimited_environment();
        
        $zip_data = file_get_contents($zip_file);
        $zip_size = strlen($zip_data);
        
        $chunk_size = 512 * 1024;
        $total_chunks = ceil($zip_size / $chunk_size);
        $session_id = wp_generate_uuid4();
        
        $this->log("[送信側] チャンク送信開始: {$total_chunks} 個");
        
        for ($i = 0; $i < $total_chunks; $i++) {
            $offset = $i * $chunk_size;
            $chunk = substr($zip_data, $offset, $chunk_size);
            $base64_chunk = base64_encode($chunk);
            
            $is_last = ($i === $total_chunks - 1);
            
            if ($i > 0 && $i % 10 === 0) {
                usleep(100000);
            }
            
            $response = wp_remote_post($endpoint, array(
                'timeout' => 300,
                'headers' => array(
                    'X-SSS-API-Key' => $api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => wp_json_encode(array(
                    'session_id' => $session_id,
                    'chunk_index' => $i,
                    'total_chunks' => $total_chunks,
                    'data' => $base64_chunk,
                    'is_last' => $is_last,
                    'table_prefix' => $table_prefix,
                    'source_url' => home_url(),
                )),
            ));
            
            if (is_wp_error($response)) {
                return array('success' => false, 'error' => $response->get_error_message());
            }
            
            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            if ($code !== 200) {
                $error_msg = $body['error'] ?? 'Unknown error (HTTP ' . $code . ')';
                $this->report_error_to_server("send_db_to_target", "HTTP {$code}: {$error_msg}");
                return array('success' => false, 'error' => $error_msg);
            }
            
            $this->log("[送信側] チャンク " . ($i + 1) . "/{$total_chunks} 送信成功");
        }
        
        return array('success' => true);
    }
    
    /**
     * DB受信ハンドラー
     */
    public function handle_db_deploy_receive($request) {
        try {
            $this->setup_unlimited_environment();
            
            $params = $request->get_json_params();
            
            $session_id = sanitize_text_field($params['session_id'] ?? '');
            $chunk_index = intval($params['chunk_index'] ?? 0);
            $total_chunks = intval($params['total_chunks'] ?? 1);
            $data = $params['data'] ?? '';
            $is_last = ($chunk_index >= $total_chunks - 1);
            $source_table_prefix = sanitize_text_field($params['table_prefix'] ?? 'wp_');
            
            if ($chunk_index === 0) {
                $this->set_deploy_receiving('database');
            }
            
            $this->log("[受信側] DBチャンク受信: " . ($chunk_index + 1) . "/{$total_chunks}");
            
            $temp_file = sys_get_temp_dir() . '/sss_db_' . $session_id . '.zip';
            $chunk_data = base64_decode($data);
            $write_result = file_put_contents($temp_file, $chunk_data, FILE_APPEND | LOCK_EX);
            
            if ($write_result === false) {
                $this->clear_deploy_receiving();
                return new WP_REST_Response(array('error' => '一時ファイル書き込み失敗'), 500);
            }
            
            if (!$is_last) {
                return new WP_REST_Response(array('success' => true, 'continue' => true), 200);
            }
            
            // 最終チャンク処理
            $this->log("[受信側] 全チャンク受信完了、インポート開始...");
            
            $extract_dir = sys_get_temp_dir() . '/sss_db_extract_' . $session_id;
            if (!is_dir($extract_dir)) {
                mkdir($extract_dir, 0755, true);
            }
            
            $zip = new ZipArchive();
            if ($zip->open($temp_file) !== true) {
                $this->clear_deploy_receiving();
                return new WP_REST_Response(array('error' => 'ZIP展開失敗'), 500);
            }
            $zip->extractTo($extract_dir);
            $zip->close();
            
            $sql_file = $extract_dir . '/database.sql';
            if (!file_exists($sql_file)) {
                $this->clear_deploy_receiving();
                return new WP_REST_Response(array('error' => 'SQLファイルが見つかりません'), 500);
            }
            
            $sql_content = file_get_contents($sql_file);
            
            global $wpdb;
            $target_prefix = $wpdb->prefix;
            
            if ($source_table_prefix !== $target_prefix) {
                $sql_content = str_replace(
                    array("`{$source_table_prefix}", "'{$source_table_prefix}"),
                    array("`{$target_prefix}", "'{$target_prefix}"),
                    $sql_content
                );
            }
            
            // 受信側設定をバックアップ
            $backup_api_key = get_option('sss_api_key');
            $backup_deploy_settings = get_option('sss_deploy_settings');
            
            // SQL実行
            $queries = $this->split_sql_queries($sql_content);
            $executed = 0;
            $errors = array();
            
            foreach ($queries as $query) {
                $query = trim($query);
                if (empty($query)) {
                    continue;
                }
                
                $result = $wpdb->query($query);
                if ($result === false) {
                    $errors[] = $wpdb->last_error;
                } else {
                    $executed++;
                }
            }
            
            // 設定復元
            if ($backup_api_key !== false) {
                delete_option('sss_api_key');
                add_option('sss_api_key', $backup_api_key, '', 'no');
            }
            if ($backup_deploy_settings !== false) {
                delete_option('sss_deploy_settings');
                add_option('sss_deploy_settings', $backup_deploy_settings, '', 'no');
            }
            
            wp_cache_flush();
            
            @unlink($sql_file);
            @rmdir($extract_dir);
            @unlink($temp_file);
            
            $this->log("★★★ DBインポート完了: {$executed} クエリ ★★★", 'success');
            $this->clear_deploy_receiving();
            
            return new WP_REST_Response(array(
                'success' => true,
                'message' => "データベースをインポートしました ({$executed} クエリ)",
            ), 200);
            
        } catch (Exception $e) {
            $this->log("[受信側] 例外発生: " . $e->getMessage(), 'error');
            $this->report_error_to_server("handle_db_deploy_receive", "Exception: " . $e->getMessage());
            $this->clear_deploy_receiving();
            return new WP_REST_Response(array('error' => 'Exception: ' . $e->getMessage()), 500);
        }
    }
    
    /**
     * SQLをクエリ単位に分割
     */
    private function split_sql_queries($sql) {
        $queries = array();
        $length = strlen($sql);
        
        if ($length > 50 * 1024 * 1024) {
            return preg_split('/;\s*\n/', $sql, -1, PREG_SPLIT_NO_EMPTY);
        }
        
        $current_query = '';
        $in_string = false;
        $string_char = '';
        
        for ($i = 0; $i < $length; $i++) {
            $char = $sql[$i];
            $prev_char = $i > 0 ? $sql[$i - 1] : '';
            
            if (($char === "'" || $char === '"') && $prev_char !== '\\') {
                if (!$in_string) {
                    $in_string = true;
                    $string_char = $char;
                } elseif ($char === $string_char) {
                    $in_string = false;
                }
            }
            
            $current_query .= $char;
            
            if ($char === ';' && !$in_string) {
                $queries[] = trim($current_query);
                $current_query = '';
            }
        }
        
        if (trim($current_query)) {
            $queries[] = trim($current_query);
        }
        
        return $queries;
    }
    
    /**
     * AJAX: メディア同期
     */
    public function ajax_sync_media() {
        check_ajax_referer('sss_deploy', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        $this->setup_unlimited_environment();
        
        $server_id = intval($_POST['server_id'] ?? 1);
        
        $deploy_settings = get_option('sss_deploy_settings', array());
        $server_key = 'server' . $server_id;
        
        if (empty($deploy_settings[$server_key]['url']) || empty($deploy_settings[$server_key]['api_key'])) {
            wp_send_json_error('ターゲットサーバーの設定が必要です');
        }
        
        $target_url = $deploy_settings[$server_key]['url'];
        $api_key = $deploy_settings[$server_key]['api_key'];
        
        $this->log("=== メディア同期開始 ===");
        $this->log("送信先: {$target_url}");
        
        $uploads_dir = wp_upload_dir();
        $uploads_base = $uploads_dir['basedir'];
        
        $zip_file = $this->create_zip($uploads_base, 'uploads');
        if (!$zip_file) {
            wp_send_json_error('メディアのZIP化に失敗しました');
        }
        
        $endpoint = $this->build_rest_url($target_url, '/sss/v1/deploy-media');
        $result = $this->send_db_to_target($endpoint, $zip_file, $api_key, '');
        
        @unlink($zip_file);
        
        if ($result['success']) {
            $this->log("★★★ メディア同期完了 ★★★", 'success');
            wp_send_json_success(array('message' => 'メディアファイルを同期しました'));
        } else {
            $this->log("メディア同期失敗: " . $result['error'], 'error');
            $this->report_error_to_server("ajax_sync_media", "Error: " . $result['error']);
            wp_send_json_error($result['error']);
        }
    }
    
    /**
     * メディア受信ハンドラー
     */
    public function handle_media_deploy_receive($request) {
        $this->setup_unlimited_environment();
        
        $params = $request->get_json_params();
        
        $session_id = sanitize_text_field($params['session_id'] ?? '');
        $chunk_index = intval($params['chunk_index'] ?? 0);
        $total_chunks = intval($params['total_chunks'] ?? 1);
        $data = $params['data'] ?? '';
        $is_last = ($chunk_index >= $total_chunks - 1);
        
        if ($chunk_index === 0) {
            $this->set_deploy_receiving('media');
            $this->log("[受信側] === メディア受信開始 ===");
        }
        
        $this->log("[受信側] メディアチャンク受信: " . ($chunk_index + 1) . "/{$total_chunks}");
        
        $temp_file = sys_get_temp_dir() . '/sss_media_' . $session_id . '.zip';
        $chunk_data = base64_decode($data);
        $write_result = file_put_contents($temp_file, $chunk_data, FILE_APPEND | LOCK_EX);
        
        if ($write_result === false) {
            $this->clear_deploy_receiving();
            return new WP_REST_Response(array('error' => 'チャンク書き込み失敗'), 500);
        }
        
        if (!$is_last) {
            return new WP_REST_Response(array('success' => true, 'continue' => true), 200);
        }
        
        // 最終チャンク処理
        $this->log("[受信側] 全チャンク受信完了、ZIP展開開始...");
        
        $uploads_dir = wp_upload_dir();
        $uploads_base = $uploads_dir['basedir'];
        $extract_to = dirname($uploads_base);
        
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        
        $result = unzip_file($temp_file, $extract_to);
        
        @unlink($temp_file);
        
        if (is_wp_error($result)) {
            $this->log("[受信側] メディア展開エラー: " . $result->get_error_message(), 'error');
            $this->report_error_to_server("handle_media_deploy_receive", "unzip_file Error: " . $result->get_error_message());
            $this->clear_deploy_receiving();
            return new WP_REST_Response(array('error' => $result->get_error_message()), 500);
        }
        
        $file_count = is_dir($uploads_base) ? $this->count_files_recursive($uploads_base) : 0;
        
        $this->log("★★★ メディアインポート完了: {$file_count} ファイル ★★★", 'success');
        $this->clear_deploy_receiving();
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'メディアをインポートしました',
            'files' => $file_count
        ), 200);
    }
}

/**
 * Release Manager - 配布サイト専用の開発者ツール
 * 配布サイトでのみ有効化され、プラグインのリリース管理を行う
 */
class SSS_Release_Manager {
    
    private static $instance = null;
    
    /**
     * 配布サイトのドメインリスト（これらのドメインでのみ有効）
     */
    private $distribution_domains = array(
        'xiscore-creative.co.jp',
        'simple-seo-migration.com',
        'localhost',
        '127.0.0.1',
    );
    
    /**
     * リリースディレクトリ
     */
    private $releases_dir;
    
    /**
     * リリースURL
     */
    private $releases_url;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // 配布サイトでのみ有効化
        if (!$this->is_distribution_site()) {
            return;
        }
        
        $upload_dir = wp_upload_dir();
        $this->releases_dir = $upload_dir['basedir'] . '/sss-releases/';
        $this->releases_url = $upload_dir['baseurl'] . '/sss-releases/';
        
        // ディレクトリ作成
        if (!is_dir($this->releases_dir)) {
            wp_mkdir_p($this->releases_dir);
        }
        
        add_action('admin_menu', array($this, 'add_developer_menu'), 30);
        add_action('wp_ajax_sss_create_release', array($this, 'ajax_create_release'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * 配布サイトかどうかをチェック
     */
    private function is_distribution_site() {
        $host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
        $host = preg_replace('/:\d+$/', '', $host); // ポート番号を除去
        
        foreach ($this->distribution_domains as $domain) {
            if ($host === $domain || strpos($host, $domain) !== false) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * 開発者メニュー追加
     */
    public function add_developer_menu() {
        add_submenu_page(
            'simple-seo-backup',
            '🔧 開発者ツール',
            '🔧 開発者ツール',
            'manage_options',
            'sss-developer',
            array($this, 'render_developer_page')
        );
    }
    
    /**
     * 管理画面スクリプト
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'sss-developer') === false) {
            return;
        }
        
        wp_enqueue_script('sss-developer', SSS_PLUGIN_URL . 'assets/developer.js', array('jquery'), SSS_VERSION, true);
        wp_localize_script('sss-developer', 'sssDev', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sss_developer'),
        ));
    }
    
    /**
     * 開発者ページ表示
     */
    public function render_developer_page() {
        // 現在のバージョン情報を取得
        $plugin_data = get_plugin_data(SSS_PLUGIN_DIR . 'simple-site-sync.php');
        $current_version = $plugin_data['Version'];
        
        // 既存リリース情報
        $info_file = $this->releases_dir . 'info.json';
        $existing_info = null;
        if (file_exists($info_file)) {
            $existing_info = json_decode(file_get_contents($info_file), true);
        }
        
        // リリース済みZIPファイル一覧
        $releases = glob($this->releases_dir . '*.zip');
        ?>
        <div class="wrap">
            <h1>🔧 Simple Site Sync 開発者ツール</h1>
            
            <div class="notice notice-warning" style="padding: 15px;">
                <p><strong>⚠️ このページは配布サイト専用です</strong></p>
                <p>ここでリリースを作成すると、他のサイトにインストールされたプラグインに更新通知が届きます。</p>
            </div>
            
            <h2>現在のバージョン</h2>
            <table class="form-table">
                <tr>
                    <th>プラグインバージョン</th>
                    <td><code style="font-size: 16px;"><?php echo esc_html($current_version); ?></code></td>
                </tr>
                <tr>
                    <th>配布済みバージョン</th>
                    <td>
                        <?php if ($existing_info): ?>
                            <code style="font-size: 16px;"><?php echo esc_html($existing_info['version']); ?></code>
                            <span style="color: #666; margin-left: 10px;">
                                (<?php echo esc_html($existing_info['last_updated'] ?? '不明'); ?>)
                            </span>
                        <?php else: ?>
                            <span style="color: #999;">未リリース</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <th>info.json URL</th>
                    <td><code><?php echo esc_url($this->releases_url . 'info.json'); ?></code></td>
                </tr>
            </table>
            
            <hr style="margin: 30px 0;">
            
            <h2>新規リリース作成</h2>
            <p class="description">
                現在のプラグインコードをZIP化し、info.json を更新します。<br>
                リリース前に <code>simple-site-sync.php</code> の <code>Version:</code> ヘッダーを更新してください。
            </p>
            
            <div style="margin-top: 20px; display: flex; gap: 15px;">
                <button type="button" class="button button-primary button-hero" id="sss-create-release">
                    <span class="dashicons dashicons-upload" style="margin-top: 5px;"></span>
                    リリース v<?php echo esc_html($current_version); ?> を作成
                </button>
                
                <button type="button" class="button button-hero" id="sss-test-error-report" style="background: #d63638; color: #fff; border-color: #d63638;">
                    <span class="dashicons dashicons-warning" style="margin-top: 5px;"></span>
                    エラー通知テスト
                </button>
            </div>
            
            <div id="sss-release-status" style="margin-top: 20px;"></div>
            
            <hr style="margin: 30px 0;">
            
            <h2>リリース履歴</h2>
            <?php if (!empty($releases)): ?>
                <table class="widefat striped" style="max-width: 800px;">
                    <thead>
                        <tr>
                            <th>ファイル名</th>
                            <th>サイズ</th>
                            <th>作成日時</th>
                            <th>アクション</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_reverse($releases) as $release): ?>
                            <?php
                            $filename = basename($release);
                            $filesize = filesize($release);
                            $filetime = filemtime($release);
                            ?>
                            <tr>
                                <td><code><?php echo esc_html($filename); ?></code></td>
                                <td><?php echo esc_html(size_format($filesize)); ?></td>
                                <td><?php echo esc_html(wp_date('Y-m-d H:i:s', $filetime)); ?></td>
                                <td>
                                    <a href="<?php echo esc_url($this->releases_url . $filename); ?>" class="button button-small" download>
                                        ダウンロード
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <p style="color: #999;">リリースはまだありません。</p>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * AJAX: リリース作成
     */
    public function ajax_create_release() {
        check_ajax_referer('sss_developer', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('権限がありません');
        }
        
        try {
            // プラグイン情報を取得
            $plugin_file = SSS_PLUGIN_DIR . 'simple-site-sync.php';
            $plugin_data = get_plugin_data($plugin_file);
            $version = $plugin_data['Version'];
            
            // ZIPファイル作成
            $zip_filename = 'simple-site-sync-' . $version . '.zip';
            $zip_path = $this->releases_dir . $zip_filename;
            
            $result = $this->create_release_zip($zip_path);
            if (!$result['success']) {
                wp_send_json_error($result['error']);
            }
            
            // ファイル権限設定 (rw-rw-r--)
            @chmod($zip_path, 0664);
            
            // info.json を更新
            $info = array(
                'name' => $plugin_data['Name'],
                'slug' => 'simple-site-sync',
                'version' => $version,
                'download_url' => $this->releases_url . $zip_filename,
                'requires' => '6.0',
                'tested' => get_bloginfo('version'),
                'requires_php' => '7.4',
                'last_updated' => wp_date('Y-m-d H:i:s'),
                'author' => $plugin_data['Author'],
                'author_homepage' => $plugin_data['AuthorURI'],
                'homepage' => $plugin_data['PluginURI'] ?: home_url(),
                'sections' => array(
                    'description' => $plugin_data['Description'],
                    'changelog' => '<h4>' . $version . '</h4><ul><li>新規リリース</li></ul>',
                ),
                'banners' => array(
                    'low' => '',
                    'high' => '',
                ),
            );
            
            $info_path = $this->releases_dir . 'info.json';
            $json_result = file_put_contents($info_path, wp_json_encode($info, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
            
            if ($json_result === false) {
                wp_send_json_error('info.json の書き込みに失敗しました');
            }
            
            // ファイル権限設定 (rw-rw-r--)
            @chmod($info_path, 0664);
            
            wp_send_json_success(array(
                'message' => 'リリース v' . $version . ' を作成しました',
                'version' => $version,
                'zip_url' => $this->releases_url . $zip_filename,
                'zip_size' => size_format(filesize($zip_path)),
            ));
            
        } catch (Exception $e) {
            wp_send_json_error('エラー: ' . $e->getMessage());
        }
    }
    
    /**
     * リリース用ZIPを作成
     */
    private function create_release_zip($zip_path) {
        $plugin_dir = SSS_PLUGIN_DIR;
        
        // 除外するパターン
        $exclude_patterns = array(
            '.git',
            '.gitignore',
            '.svn',
            'node_modules',
            '.DS_Store',
            'Thumbs.db',
            '*.log',
            'debug.log',
            '.idea',
            '.vscode',
            'tests',
            'phpunit.xml',
            'composer.lock',
        );
        
        $zip = new ZipArchive();
        if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            return array('success' => false, 'error' => 'ZIPファイルを作成できません');
        }
        
        $plugin_dir = realpath($plugin_dir);
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        $file_count = 0;
        foreach ($files as $file) {
            $file_path = $file->getRealPath();
            $relative_path = substr($file_path, strlen($plugin_dir) + 1);
            
            // 除外チェック
            $exclude = false;
            foreach ($exclude_patterns as $pattern) {
                if (strpos($pattern, '*') !== false) {
                    $regex = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
                    if (preg_match($regex, basename($relative_path))) {
                        $exclude = true;
                        break;
                    }
                } else {
                    if (strpos($relative_path, $pattern) === 0 || 
                        strpos($relative_path, '/' . $pattern) !== false ||
                        basename($relative_path) === $pattern) {
                        $exclude = true;
                        break;
                    }
                }
            }
            
            if ($exclude) {
                continue;
            }
            
            // simple-site-sync/ フォルダに入れる
            $zip_path_internal = 'simple-site-sync/' . $relative_path;
            
            if ($file->isDir()) {
                $zip->addEmptyDir($zip_path_internal);
            } else {
                $zip->addFile($file_path, $zip_path_internal);
                $file_count++;
            }
        }
        
        $zip->close();
        
        return array('success' => true, 'files' => $file_count);
    }
}

// プラグイン初期化
Simple_Site_Sync::get_instance();

// 自動更新チェッカー初期化
$sss_update_checker = PucFactory::buildUpdateChecker(
    SSS_UPDATE_SERVER . '/wp-content/uploads/sss-releases/info.json',
    __FILE__,
    'simple-site-sync'
);

// Release Manager 初期化（配布サイトでのみ有効）
SSS_Release_Manager::get_instance();
